use bevy::prelude::*;
use bevy_egui::{
    egui::{self},
    EguiContexts,
};

use crate::{
    game::{
        controls::placer::TileClickEvent,
        map::Map,
        metrics::MetricsRate,
        resources::DefaultInfoMarker,
        tiles::{assets::TilePreviews, TileType},
    },
    i18n::{Localization, Translate},
    settings::ActiveSettingsBank,
};

use super::{popup_window, tile_info_display, TileInfoWindow};

pub fn default_tile_clicked(
    mut tile_click_event: EventReader<TileClickEvent>,
    tiles: Query<&TileType, With<DefaultInfoMarker>>,
    map: Res<Map>,
    mut market_window: ResMut<TileInfoWindow>,
) {
    if tile_click_event.is_empty() {
        return;
    }

    let click = tile_click_event.read().last().unwrap();

    if let Some(Ok(_)) = map.get(click.tile_pos).map(|e| tiles.get(e)) {
        market_window.open = true;
        market_window.selected_tile = Some(click.tile_pos);
    }
}

pub fn render_default_tile_popup(
    mut contexts: EguiContexts,
    mut tiles: Query<(&MetricsRate, &TileType), With<DefaultInfoMarker>>,
    map: Res<Map>,
    mut tile_window: ResMut<TileInfoWindow>,
    localization: Res<Localization>,
    previews: Res<TilePreviews>,
    settings: Res<ActiveSettingsBank>,
) {
    // no market selected -> don't render Window
    if !tile_window.open {
        return;
    }

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let Some(sel_tile_coord) = tile_window.selected_tile else {
        return;
    };

    // guaranteed to be Some because it is checked before
    let Some(Ok((sel_rate, sel_tile))) = map.get(sel_tile_coord).map(|e| tiles.get_mut(e)) else {
        return;
    };

    popup_window(
        egui,
        &mut tile_window.open,
        &localization.translate("tile-popup"),
        |ui| {
            tile_info_display(
                ui,
                sel_tile,
                &sel_tile_coord,
                &previews,
                &localization,
                &settings,
            );

            ui.heading(localization.translate("tile-popup-amount-rate"));
            egui::Grid::new("tile-popup-info-stats")
                .num_columns(3)
                .spacing([40.0, 4.0])
                .striped(true)
                .show(ui, |ui| {
                    ui.label(format!(
                        "{:.2} {}",
                        sel_rate.food,
                        localization.translate("game-metrics-food")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        sel_rate.materials,
                        localization.translate("game-metrics-materials")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        sel_rate.money,
                        localization.translate("game-metrics-money")
                    ));
                    ui.end_row();
                });
        },
    );
}
