#[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
use bevy::app::AppExit;
use bevy::prelude::*;
use bevy_egui::{
    egui::{self, OpenUrl},
    EguiContexts,
};

use crate::{
    define_asset_collection,
    game::{
        controls::{cam::PanOrbitCam, KeyboardEvent},
        visuals::blur::BlurSettings,
        GamePauseState,
    },
    i18n::Localization,
    prelude::*,
    settings::ActiveSettingsBank,
    ui::{
        commons::ShowGameUi,
        is_typing_egui,
        theme::{
            BACKGROUND_COLOR, BIG_TEXT_SIZE, BUTTON_COLOR, BUTTON_TEXT_COLOR, MARGIN_SIZE,
            OUTER_MARGIN, PARAGRAPH_FONT_NAME, PRIMARY_BUTTON_SIZE, PRIMARY_BUTTON_TEXT_SIZE,
            SPACE_SIZE, TITLE_FONT_NAME, TITLE_TEXT_COLOR,
        },
        UiClickEvent, WEBSITE_BASE,
    },
    AppState,
};

use super::layout::{init_layout_manager, LayoutManager};
use super::metrics::StatisticsViewState;
use super::settings::InGameSettingsState;
use super::IconButtonCard;

const BLUR_STRENGTH: f32 = 100.0;

pub struct PauseUiPlugin;
impl Plugin for PauseUiPlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<PauseUiState>();

        app.register_asset_collection::<PauseButtonIcon>();
        app.add_systems(
            Update,
            toggle_game_playpause
                .run_if(in_state(AppState::InGame))
                .run_if(in_state(GamePauseState::Running).or(in_state(PauseUiState::Shown))),
        );
        app.add_systems(
            Update,
            ui_pause_button
                .after(init_layout_manager)
                .run_if(in_state(AppState::InGame))
                .run_if(in_state(GamePauseState::Running))
                .run_if(in_state(ShowGameUi::On)),
        );

        app.add_systems(
            OnEnter(GamePauseState::Paused),
            blur_game_view.run_if(in_state(AppState::InGame)),
        );

        app.add_systems(
            OnExit(GamePauseState::Paused),
            unblur_game_view.run_if(in_state(AppState::InGame)),
        );
    }
}

#[derive(SubStates, Default, Debug, Clone, Hash, Eq, PartialEq)]
#[source(GamePauseState = GamePauseState::Paused)]
pub enum PauseUiState {
    /// the pause screen is visible and the game is paused
    Shown,
    /// the game is paused, and the background blurred,
    /// however the pause screen is not visible
    #[default]
    Hidden,
}

fn toggle_game_playpause(
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    curr: Res<State<GamePauseState>>,
    mut next: ResMut<NextState<GamePauseState>>,
    mut next_ui: ResMut<NextState<PauseUiState>>,
    mut kb_event: EventWriter<KeyboardEvent>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }
    if settings
        .keybindings
        .menu
        .any(|code| input.just_pressed(code))
    {
        match curr.get() {
            GamePauseState::Paused => next.set(GamePauseState::Running),
            GamePauseState::Running => {
                next.set(GamePauseState::Paused);
                next_ui.set(PauseUiState::Shown);
            }
        };
        kb_event.send(KeyboardEvent);
    }
}

#[allow(clippy::too_many_arguments)]
pub fn ui_pause_screen<const SHOW_STATS: bool>(
    mut contexts: EguiContexts,
    windows: Query<&Window>,
    localization: Res<Localization>,
    mut playpause: ResMut<NextState<GamePauseState>>,
    mut appstate: ResMut<NextState<AppState>>,
    mut pauseui: ResMut<NextState<PauseUiState>>,
    mut settingsstate: ResMut<NextState<InGameSettingsState>>,
    mut viewstats: ResMut<NextState<StatisticsViewState>>,
    mut exit: EventWriter<AppExit>,
    mut click_event: EventWriter<UiClickEvent>,
    settings: Res<ActiveSettingsBank>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let my_frame = egui::Frame {
        fill: BACKGROUND_COLOR.gamma_multiply(0.7),
        inner_margin: egui::Margin::same((window.y * MARGIN_SIZE) as i8),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };

    // center panel
    egui::CentralPanel::default()
        .frame(my_frame)
        // .show_separator_line(false) // doesn't work
        .show(egui, |_ui| {});

    egui::Area::new("game-pause-overlay".into())
        .anchor(egui::Align2::CENTER_CENTER, [0.0, 0.0])
        .show(egui, |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().interaction.selectable_labels = false;

                ui.label(
                    egui::RichText::new(localization.translate("paused"))
                        .font(egui::FontId::new(
                            window.y * BIG_TEXT_SIZE,
                            egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                        ))
                        .color(TITLE_TEXT_COLOR),
                );

                ui.style_mut().text_styles.insert(
                    egui::TextStyle::Button,
                    egui::FontId::new(
                        window.y * PRIMARY_BUTTON_TEXT_SIZE / 2.,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ),
                );

                // return to game
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("resume-game"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    playpause.set(GamePauseState::Running);
                    click_event.send(UiClickEvent);
                }
                ui.add_space(window.y * SPACE_SIZE);

                if SHOW_STATS {
                    // view stats
                    if ui
                        .add_sized(
                            (window * PRIMARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                egui::RichText::new(localization.translate("view-stats"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                    {
                        viewstats.set(StatisticsViewState::FromPaused);
                        pauseui.set(PauseUiState::Hidden);
                        click_event.send(UiClickEvent);
                    }
                    ui.add_space(window.y * SPACE_SIZE);
                }

                // view settings
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("view-settings"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    click_event.send(UiClickEvent);
                    pauseui.set(PauseUiState::Hidden);
                    settingsstate.set(InGameSettingsState::Shown);
                }
                ui.add_space(window.y * SPACE_SIZE);

                // show help
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("show-help"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    ui.ctx().open_url(OpenUrl::new_tab(format!(
                        "{}/{}/book/",
                        WEBSITE_BASE,
                        settings.lang.get_book_id()
                    )));
                }
                ui.add_space(window.y * SPACE_SIZE);

                // return to main menu
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("exit-to-menu"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    click_event.send(UiClickEvent);
                    appstate.set(AppState::AppMenu);
                }
                ui.add_space(window.y * SPACE_SIZE);

                // close app
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("exit-to-os"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    click_event.send(UiClickEvent);
                    exit.send(AppExit::Success);
                }
            });
        });
}

define_asset_collection!(
    PauseButtonIcon,
    !pause_icon : Image = "icons/buttons/pause.png",
    err : "buttton-icon-missing" "Button icon missing." "button-icon-missing-desc" "You can not use a couple of buttons without the icons"
);

pub fn ui_pause_button(
    windows: Query<&Window>,
    mut layout: ResMut<LayoutManager>,
    mut contexts: EguiContexts,
    mut pause: ResMut<NextState<GamePauseState>>,
    mut pause_ui: ResMut<NextState<PauseUiState>>,
    mut click_event: EventWriter<UiClickEvent>,
    icon: Res<PauseButtonIcon>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.width() == 0.0 || window.height() == 0.0 {
        return;
    }

    let icon_texture = contexts.add_image(icon.pause_icon.clone_weak());
    let icon_size = window.height() * BIG_TEXT_SIZE / 4.0;

    let icon = IconButtonCard {
        icon_size,
        texture: icon_texture,
    };

    let cell = if let Some(cell) = layout.allocate_margin(
        super::layout::LayoutOrigin::RightTop,
        super::layout::LayoutSizeRequest::Fixed(icon.get_size()),
        super::layout::LayoutSizeRequest::Fixed(icon.get_size()),
        super::layout::LayoutDirection::Horizontal,
        OUTER_MARGIN,
    ) {
        cell
    } else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    egui::Area::new("pause-button".into())
        .pivot(egui::Align2::RIGHT_TOP)
        .fixed_pos(cell.rect.right_top())
        .show(egui, |ui| {
            if ui.add(icon).clicked() {
                click_event.send(UiClickEvent);
                pause.set(GamePauseState::Paused);
                pause_ui.set(PauseUiState::Shown);
            }
        });
}

pub fn blur_game_view(
    mut commands: Commands,
    windows: Query<&Window>,
    mut cams: Query<Entity, With<PanOrbitCam>>,
) {
    let Ok(_window) = windows.get_single() else {
        return;
    };
    let Ok(cam) = cams.get_single_mut() else {
        return;
    };
    commands
        .entity(cam)
        .insert(BlurSettings::new(BLUR_STRENGTH));
}

pub fn unblur_game_view(
    mut commands: Commands,
    windows: Query<&Window>,
    mut cams: Query<Entity, (With<PanOrbitCam>, With<BlurSettings>)>,
) {
    let Ok(_window) = windows.get_single() else {
        return;
    };
    let Ok(cam) = cams.get_single_mut() else {
        return;
    };
    commands.entity(cam).remove::<BlurSettings>();
}
