use bevy::{
    app::{Plugin, Update},
    prelude::{in_state, IntoSystemConfigs, Query, ResMut, Resource},
    window::Window,
};
use bevy_egui::egui::{Pos2, Rect, Vec2};

use crate::AppState;

pub struct LayoutManagerPlugin;
impl Plugin for LayoutManagerPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.init_resource::<LayoutManager>();
        app.add_systems(
            Update,
            init_layout_manager.run_if(in_state(AppState::InGame)),
        );
    }
}

/// Allocated screen area
#[derive(Clone)]
pub struct Cell {
    /// the allocated area itself
    pub rect: Rect,
}

/// Size request wrapper (1D)
#[allow(dead_code)]
pub enum LayoutSizeRequest {
    /// request fixed size
    Fixed(f32),
    /// fill all available space
    Filled,
}
impl LayoutSizeRequest {
    fn map<M>(&self, func: M) -> Self
    where
        M: Fn(f32) -> f32,
    {
        match self {
            LayoutSizeRequest::Filled => LayoutSizeRequest::Filled,
            LayoutSizeRequest::Fixed(f) => LayoutSizeRequest::Fixed(func(*f)),
        }
    }
}

/// locations from where a layout area can grow
#[derive(Clone)]
#[allow(dead_code)]
pub enum LayoutOrigin {
    LeftTop,
    RightTop,
    LeftBottom,
    RightBottom,
}
/// in which direction to move new elements
#[derive(PartialEq)]
#[allow(dead_code)]
pub enum LayoutDirection {
    /// move elements along x axis
    Horizontal,
    /// move elements along y axis
    Vertical,
}

#[derive(Resource, Default)]
pub struct LayoutManager {
    /// all known allocated areas
    cells: Vec<Cell>,
    /// window dimensions
    window: Option<Vec2>,
}
impl LayoutManager {
    /// try to allocate space for a widget to be placed in
    /// the area will grow from the origin
    /// returns None if allocation failed
    pub fn allocate(
        &mut self,
        origin: LayoutOrigin,
        width: LayoutSizeRequest,
        height: LayoutSizeRequest,
        direction: LayoutDirection,
    ) -> Option<Cell> {
        let window = if let Some(window) = &self.window {
            window
        } else {
            // No window to allocate space in
            return None;
        };

        // very rudimentary layout algorithm
        // doesn't implement edge cases that we won't run into
        // our ingame UI aligns to the screen edges
        // and normally doesn't extend above that
        let mut from_left = 0.0;
        let mut from_right = window.x;
        let mut from_top = 0.0;
        let mut from_bottom = window.y;

        // check for areas that were already allocated
        for cell in self.cells.iter() {
            if let LayoutSizeRequest::Fixed(h) = height {
                let self_y = match origin {
                    LayoutOrigin::LeftTop | LayoutOrigin::RightTop => (0.0, h),
                    LayoutOrigin::LeftBottom | LayoutOrigin::RightBottom => {
                        (window.y - h, window.y)
                    }
                };
                // compare the size of the areas that overlap along the y axis
                if max_f32(cell.rect.min.y, self_y.0) <= min_f32(cell.rect.max.y, self_y.1) {
                    if cell.rect.max.x > from_left
                        && cell.rect.min.x == from_left
                        && direction == LayoutDirection::Horizontal
                    {
                        // found a cell located next to the last cell from the left
                        // which means that this one is the last cell from the left
                        from_left = cell.rect.max.x;
                    }
                    if cell.rect.min.x < from_right
                        && cell.rect.max.x == from_right
                        && direction == LayoutDirection::Horizontal
                    {
                        // found a cell located next to the last cell from the right
                        // which means that this one is the last cell from the right
                        from_right = cell.rect.min.x;
                    }
                }
            }

            if let LayoutSizeRequest::Fixed(w) = width {
                let self_x = match origin {
                    LayoutOrigin::LeftBottom | LayoutOrigin::LeftTop => (0.0, w),
                    LayoutOrigin::RightBottom | LayoutOrigin::RightTop => (window.x - w, window.x),
                };
                // compare the size of the areas that overlap along the x axis
                if max_f32(cell.rect.min.x, self_x.0) <= min_f32(cell.rect.max.x, self_x.1)
                    && direction == LayoutDirection::Vertical
                {
                    if cell.rect.max.y > from_top && cell.rect.min.y == from_top {
                        // found a cell located next to the last cell from the top
                        // which means that this one is the last cell from the top
                        from_top = cell.rect.max.y;
                    }
                    if cell.rect.min.y < from_bottom
                        && cell.rect.max.y == from_bottom
                        && direction == LayoutDirection::Vertical
                    {
                        // found a cell located next to the last cell from the bottom
                        // which means that this one is the last cell from the bottom
                        from_bottom = cell.rect.min.y;
                    }
                }
            }
        }

        let cell = match (width, height) {
            (LayoutSizeRequest::Filled, LayoutSizeRequest::Filled) => Some(Cell {
                rect: Rect::from_min_max(
                    Pos2::new(from_left, from_top),
                    Pos2::new(from_right, from_bottom),
                ),
            }),
            (LayoutSizeRequest::Fixed(w), LayoutSizeRequest::Fixed(h)) => {
                if (from_right - from_left < w) || (from_bottom - from_top < h) {
                    // requested area cannot be allocated
                    // no space left
                    return None;
                }
                Some(match origin {
                    LayoutOrigin::RightTop => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_right - w, from_top),
                            Pos2::new(from_right, from_top + h),
                        ),
                    },
                    LayoutOrigin::RightBottom => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_right - w, from_bottom - h),
                            Pos2::new(from_right, from_bottom),
                        ),
                    },
                    LayoutOrigin::LeftTop => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_left, from_top),
                            Pos2::new(from_left + w, from_top + h),
                        ),
                    },
                    LayoutOrigin::LeftBottom => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_left, from_bottom - h),
                            Pos2::new(from_left + w, from_bottom),
                        ),
                    },
                })
            }
            (LayoutSizeRequest::Fixed(w), LayoutSizeRequest::Filled) => {
                if from_right - from_left < w {
                    // requested area cannot be allocated
                    // no space left
                    return None;
                }
                Some(match origin {
                    LayoutOrigin::RightTop => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_right - w, from_top),
                            Pos2::new(from_right, from_bottom),
                        ),
                    },
                    LayoutOrigin::RightBottom => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_right - w, from_top),
                            Pos2::new(from_right, from_bottom),
                        ),
                    },
                    LayoutOrigin::LeftTop => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_left, from_top),
                            Pos2::new(from_left + w, from_bottom),
                        ),
                    },
                    LayoutOrigin::LeftBottom => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_left, from_top),
                            Pos2::new(from_left + w, from_bottom),
                        ),
                    },
                })
            }
            (LayoutSizeRequest::Filled, LayoutSizeRequest::Fixed(h)) => {
                if from_bottom - from_top < h {
                    // requested area cannot be allocated
                    // no space left
                    return None;
                }
                Some(match origin {
                    LayoutOrigin::RightTop => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_left, from_top),
                            Pos2::new(from_right, from_top + h),
                        ),
                    },
                    LayoutOrigin::RightBottom => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_left, from_bottom - h),
                            Pos2::new(from_right, from_bottom),
                        ),
                    },
                    LayoutOrigin::LeftTop => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_left, from_top),
                            Pos2::new(from_right, from_top + h),
                        ),
                    },
                    LayoutOrigin::LeftBottom => Cell {
                        rect: Rect::from_min_max(
                            Pos2::new(from_left, from_bottom - h),
                            Pos2::new(from_right, from_bottom),
                        ),
                    },
                })
            }
        };

        if let Some(cell) = &cell {
            self.cells.push(cell.clone());
        }

        cell
    }
    /// same as `.allocate` but with a margin,
    /// that increases the requested width/height
    /// and moves the returned cell
    pub fn allocate_margin(
        &mut self,
        origin: LayoutOrigin,
        width: LayoutSizeRequest,
        height: LayoutSizeRequest,
        direction: LayoutDirection,
        margin: f32,
    ) -> Option<Cell> {
        let raw = self.allocate(
            origin,
            width.map(|w| w + margin * 2.0),
            height.map(|h| h + margin * 2.0),
            direction,
        );
        raw.map(|c| {
            let mut cell = c.clone();
            cell.rect = cell.rect.shrink(margin);
            cell
        })
    }
}

pub fn max_f32(a: f32, b: f32) -> f32 {
    if a < b {
        return b;
    }
    a
}
pub fn min_f32(a: f32, b: f32) -> f32 {
    if a > b {
        return b;
    }
    a
}

/// updated the layout manager window dimensions
/// and clears the list of known cells
/// run all you UI systems after this (in correct order)
pub fn init_layout_manager(mut mgr: ResMut<LayoutManager>, windows: Query<&Window>) {
    mgr.cells.clear();

    let Ok(window) = windows.get_single() else {
        // no single window found
        // so area allocation won't be possible
        mgr.window = None;
        return;
    };
    let window = Vec2::new(window.width(), window.height());

    if window.y == 0.0 || window.x == 0.0 {
        // window has no size
        // so area allocation would be useless
        mgr.window = None;
        return;
    }

    mgr.window = Some(window);
}
