use bevy_egui::egui::{
    self, load::SizedTexture, Color32, CornerRadius, Frame, Margin, Stroke, Widget,
};

use crate::ui::theme::{NORMAL_TEXT_COLOR, PARAGRAPH_FONT_NAME};

use super::super::theme::{
    INV_SLOT_BG, INV_SLOT_INNER_MARGIN, INV_SLOT_OUTER_MARGIN, INV_SLOT_RADIUS, INV_SLOT_STROKE,
    INV_SLOT_STROKE_HOVER, INV_SLOT_STROKE_HOVER_SELECTED, INV_SLOT_STROKE_SELECTED,
    INV_SLOT_STROKE_WIDTH, INV_SLOT_STROKE_WIDTH_SELECTED,
};

pub struct InventorySlot {
    /// width/height of the box
    pub size: f32,
    /// the tile in this slot
    /// the textureID should be the registered preview image
    /// and the second argument the translated tile name
    /// that is shown in tooltips
    pub item: Option<(egui::TextureId, String)>,
    /// if this slot is focused
    /// changes the outline
    pub selected: bool,
    /// slot id
    pub id: String,
    /// displays the id at the top right
    pub show_id: bool,
    /// color to tint the item image with
    pub tint: Color32,
}
impl Default for InventorySlot {
    fn default() -> Self {
        Self {
            size: 64.0,
            item: None,
            selected: false,
            id: String::new(),
            show_id: false,
            tint: Color32::WHITE,
        }
    }
}
impl Widget for InventorySlot {
    fn ui(self, ui: &mut egui::Ui) -> egui::Response {
        let inner_margin = INV_SLOT_INNER_MARGIN * self.size;
        let outer_margin = INV_SLOT_OUTER_MARGIN * self.size;

        let selected_stroke_width = outer_margin * INV_SLOT_STROKE_WIDTH_SELECTED;
        let default_stroke_width = outer_margin * INV_SLOT_STROKE_WIDTH;

        let stroke_width = if self.selected {
            selected_stroke_width
        } else {
            default_stroke_width
        };

        let frame = Frame {
            fill: INV_SLOT_BG,
            stroke: Stroke::new(
                stroke_width,
                if self.selected {
                    INV_SLOT_STROKE_SELECTED
                } else {
                    INV_SLOT_STROKE
                },
            ),
            outer_margin: Margin::same(outer_margin as i8),
            // use larger inner margin if the slot is not selected,
            // should keep the size of the icon fairly consistent
            inner_margin: Margin::same(
                (inner_margin
                    + if !self.selected {
                        selected_stroke_width - default_stroke_width
                    } else {
                        0.0
                    }) as i8,
            ),
            corner_radius: CornerRadius::same(INV_SLOT_RADIUS as u8),
            ..Default::default()
        };
        // manually handle frame painting
        let mut prep = frame.begin(ui);
        {
            let ui = &mut prep.content_ui;
            // frame content
            let available_space = (
                self.size - ((inner_margin as i8) * 2) as f32 - ((outer_margin as i8) * 2) as f32
                //- ((selected_stroke_width as i8) * 2) as f32
            )
                .clamp(0.0, self.size);
            if let Some((texture, _)) = self.item {
                ui.add(
                    egui::Image::new(SizedTexture::new(
                        texture,
                        [available_space, available_space],
                    ))
                    .tint(self.tint),
                );
            } else {
                ui.allocate_space(egui::Vec2::new(available_space, available_space));
            }
        };
        // manually allocate space
        // because we want to adjust the frame on hover before painting
        let resp = prep.allocate_space(ui);
        // the frame only supports hover by default
        let resp = resp.interact(egui::Sense::click());
        if resp.hovered() {
            // if the slot has an item
            // show a tooltip with the item name on hover
            if let Some((_, name)) = self.item {
                if ui.ui_contains_pointer() {
                    egui::show_tooltip_at_pointer(
                        ui.ctx(),
                        ui.layer_id(),
                        format!("inventory-{}-{}-tooltip", name, self.id).into(),
                        |ui| {
                            ui.label(name);
                        },
                    );
                }
            }

            // change stroke color to indicate hover
            prep.frame.stroke.color = if self.selected {
                INV_SLOT_STROKE_HOVER_SELECTED
            } else {
                INV_SLOT_STROKE_HOVER
            };
        }

        // paint ID in top right corner
        if self.show_id {
            let top_right = resp.rect.right_top()
                - egui::Vec2::new(
                    outer_margin + selected_stroke_width + inner_margin / 2.0,
                    -(outer_margin + selected_stroke_width + inner_margin / 2.0),
                );
            ui.painter().text(
                top_right,
                egui::Align2::RIGHT_TOP,
                self.id.to_string(),
                egui::FontId::new(12.0, egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into())),
                NORMAL_TEXT_COLOR,
            );
        }

        // manually paint the frame
        prep.paint(ui);

        resp
    }
}
