use bevy::prelude::*;
use bevy_egui::{
    egui::{
        self, load::SizedTexture, Align2, Area, FontId, Frame, Id, RichText, Sense, Shadow, Vec2,
    },
    EguiContexts,
};

use crate::{
    define_asset_collection,
    game::{
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
            },
            challenge::ChallengeGameState,
            zen::ZenGameState,
        },
        hand::Hand,
        metrics::Metrics,
        round::{HandRedrawCost, ManualRedrawEvent, NewHandTimer},
        tiles::assets::TilePreviews,
        time::{Game, GameTime},
        GamePauseState,
    },
    i18n::Localization,
    prelude::*,
    settings::ActiveSettingsBank,
    ui::{
        commons::ShowGameUi,
        is_typing_egui,
        theme::{
            BOLD_FONT_NAME, BUTTON_COLOR, INV_SLOT_BG, INV_SLOT_RADIUS, INV_SLOT_SIZE,
            INV_SLOT_STROKE, OUTER_MARGIN, PARAGRAPH_FONT_NAME, REDRAW_AFFORD_COLOR, REDRAW_COLOR,
            REDRAW_FREE_ANIMATION_COLOR, REDRAW_FREE_ANIMATION_DISTANCE,
            REDRAW_FREE_ANIMATION_PERIOD, REDRAW_PAID_ICON_OPACITY, REDRAW_TIMEOUT_OVERLAY_OPACITY,
            SMALL_TEXT_SIZE,
        },
        UiClickEvent,
    },
};

use super::{
    creative_inventory::dynamic_layout,
    inventory::InventorySlot,
    layout::{init_layout_manager, LayoutManager},
    pause::ui_pause_button,
    progress_rect::ProgressRect,
};

macro_rules! define_hand_ui_plugin {
    ($({
        // state in which to display the redrawbutton, hand
        // and listen to keybindings
        state: $state:expr$(,)?
    }),*) => {
        pub struct HandUiPlugin;
        impl Plugin for HandUiPlugin {
            fn build(&self, app: &mut App) {
                app.register_asset_collection::<RedrawButtonAssets>();

                use crate::prelude::never;

                app.add_systems(
                    Update,
                    render_redraw_button
                        .after(init_layout_manager)
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(in_state(ShowGameUi::On)),
                );
                app.add_systems(
                    Update,
                    render_player_hand
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(in_state(ShowGameUi::On))
                        .after(render_redraw_button)
                        .after(ui_pause_button),
                );

                app.add_systems(
                    Update,
                    select_slot_using_keybindings
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running)),
                );
                app.add_systems(
                    Update,
                    redraw_keybinding
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running)),
                );
            }
        }
    }
}
define_hand_ui_plugin![
    {
        state: ChallengeGameState::InRound
    },
    {
        state: CampaignChallengeState::InRound
    },
    {
        state: CampaignWettbewerbState::InRound
    },
    {
        state: ZenGameState::InRound
    }
];

pub fn render_player_hand(
    mut contexts: EguiContexts,
    localization: Res<Localization>,
    mut hand: ResMut<Hand>,
    mut click_event: EventWriter<UiClickEvent>,
    mut layout_mgr: ResMut<LayoutManager>,
    previews: Res<TilePreviews>,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let screen_size = egui.input(|i| i.screen_rect);

    let slot_size = screen_size.width().min(screen_size.height()) * INV_SLOT_SIZE;

    let is_landscape = screen_size.width() > screen_size.height();
    let (area, max) = if is_landscape {
        // landscape mode
        // display inventory at bottom
        let Some(cell) = layout_mgr.allocate_margin(
            super::layout::LayoutOrigin::LeftBottom,
            super::layout::LayoutSizeRequest::Filled,
            super::layout::LayoutSizeRequest::Fixed(slot_size),
            super::layout::LayoutDirection::Horizontal,
            OUTER_MARGIN,
        ) else {
            return;
        };
        let area = egui::Area::new("cards-hand-landscape".into())
            .pivot(egui::Align2::CENTER_BOTTOM)
            .fixed_pos(cell.rect.center_bottom());
        (area, cell.rect.width())
    } else {
        // portrait mode
        // display inventory on the right
        let Some(cell) = layout_mgr.allocate_margin(
            super::layout::LayoutOrigin::RightBottom,
            super::layout::LayoutSizeRequest::Fixed(slot_size),
            super::layout::LayoutSizeRequest::Filled,
            super::layout::LayoutDirection::Vertical,
            OUTER_MARGIN,
        ) else {
            return;
        };
        let area = egui::Area::new("cards-hand-portrait".into())
            .pivot(egui::Align2::RIGHT_CENTER)
            .fixed_pos(cell.rect.right_center());
        (area, cell.rect.height())
    };
    let slot_size = slot_size.min(max);

    area.show(egui, |ui| {
        dynamic_layout(ui, screen_size, |ui| {
            let mut selected = None;
            for (i, tile) in hand.items.iter().enumerate() {
                if ui
                    .add(InventorySlot {
                        size: slot_size,
                        selected: Some(i) == hand.selected,
                        id: (i + 1).to_string(),
                        show_id: true,
                        item: Some((
                            previews
                                .get_egui_handle(*tile)
                                .expect("Every tile has a preview"),
                            localization.translate(tile.get_translation_id()),
                        )),
                        ..Default::default()
                    })
                    .clicked()
                {
                    selected = Some(i);
                    click_event.send(UiClickEvent);
                }
            }
            if selected.is_some() {
                hand.selected = selected;
            }
        });
    });
}

define_asset_collection!(
    RedrawButtonAssets,
    !redraw : Image = "icons/buttons/redraw.png",
    !resources : Image = "icons/buttons/resources.png",
    !hourglass : Image = "icons/buttons/hourglass.png",
);

#[allow(clippy::too_many_arguments)]
pub fn render_redraw_button(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut redraw_writer: EventWriter<ManualRedrawEvent>,
    redraw_price: Res<HandRedrawCost>,
    hand_interval: Res<NewHandTimer>,
    metrics: Res<Metrics>,
    mut layout: ResMut<LayoutManager>,
    icons: Res<RedrawButtonAssets>,
    time: Res<Time<Virtual>>,
    gtime: Res<Time<Game>>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };
    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.height() == 0.0 || window.width() == 0.0 {
        return;
    }

    let button_size = window.width().min(window.height()) * INV_SLOT_SIZE;

    let cell = if let Some(cell) = layout.allocate_margin(
        super::layout::LayoutOrigin::RightBottom,
        super::layout::LayoutSizeRequest::Fixed(button_size),
        super::layout::LayoutSizeRequest::Fixed(button_size),
        super::layout::LayoutDirection::Horizontal,
        OUTER_MARGIN,
    ) {
        cell
    } else {
        return;
    };

    let redraw_icon = contexts.add_image(icons.redraw.clone_weak());
    let resources_icon = contexts.add_image(icons.resources.clone_weak());
    let hourglass_icon = contexts.add_image(icons.hourglass.clone_weak());

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    Area::new(Id::new("manual-redraw-button"))
        .pivot(egui::Align2::RIGHT_BOTTOM)
        .fixed_pos(cell.rect.right_bottom())
        .show(egui, |ui| {
            // begin drawing
            let resp = Frame::NONE
                .corner_radius(INV_SLOT_RADIUS)
                .fill(INV_SLOT_BG)
                .stroke(egui::Stroke::new(
                    (OUTER_MARGIN / 2.0) as i8,
                    INV_SLOT_STROKE,
                ))
                // only show shadow when redraw is free
                .shadow(if (*redraw_price).is_none() {
                    let half_period = REDRAW_FREE_ANIMATION_PERIOD / 2.0;
                    let state = REDRAW_FREE_ANIMATION_DISTANCE
                        * (time.elapsed_secs() % REDRAW_FREE_ANIMATION_PERIOD - half_period).abs()
                        / half_period;
                    Shadow {
                        offset: [0, 0],
                        blur: state as u8,
                        spread: state as u8,
                        color: REDRAW_FREE_ANIMATION_COLOR,
                    }
                } else {
                    Shadow::NONE
                })
                .show(ui, |ui| {
                    let (rect, _) =
                        ui.allocate_exact_size(Vec2::new(button_size, button_size), Sense::click());

                    let pos = rect.center();

                    // always show the free-redraw icon
                    // when the redraw isn't free, an overlay will be shown above it,
                    // to indicate that it is locked behind "a paywall"
                    // always showing the icon indicates that pressing the button redraws cards
                    egui::Area::new("redraw-icon".into())
                        .order(egui::Order::Tooltip)
                        .interactable(false)
                        .constrain(true)
                        .fixed_pos(pos)
                        .pivot(Align2::CENTER_CENTER)
                        .show(ui.ctx(), |ui| {
                            ui.add(
                                egui::widgets::Image::new(SizedTexture::new(
                                    redraw_icon,
                                    egui::Vec2::new(button_size * 0.4, button_size * 0.4),
                                ))
                                .tint(
                                    BUTTON_COLOR.gamma_multiply(if (*redraw_price).is_some() {
                                        REDRAW_PAID_ICON_OPACITY
                                    } else {
                                        1.0
                                    }),
                                ),
                            );

                            if (*redraw_price).is_some() {
                                // only show the progress overlay when redrawing isn't free
                                ui.scope(|ui| {
                                    ui.set_opacity(REDRAW_TIMEOUT_OVERLAY_OPACITY);
                                    ProgressRect {
                                        // make sure the progress overlay does not overlap with the outline
                                        size: button_size,
                                        rounding: INV_SLOT_RADIUS,
                                        // adjust time with sub secs of game time
                                        progress: 1.0
                                            - ((hand_interval.elapsed().as_secs_f32()
                                                + gtime.sub_secs())
                                                / hand_interval.duration().as_secs_f32()),
                                        rounding_segments: 10,
                                        fill: INV_SLOT_BG,
                                    }
                                    .paint(ui.painter(), |pos| pos + rect.center().to_vec2());
                                });
                            }
                        });

                    egui::Area::new("redraw-text".into())
                        .order(egui::Order::Tooltip)
                        .interactable(false)
                        .constrain(true)
                        .fixed_pos(pos)
                        .pivot(Align2::CENTER_CENTER)
                        .show(ui.ctx(), |ui| {
                            ui.style_mut().interaction.selectable_labels = false;
                            ui.style_mut().wrap_mode = Some(egui::TextWrapMode::Extend);
                            if let Some(costs) = &**redraw_price {
                                let money = costs.money.as_human_readable();
                                let materials = costs.materials.as_human_readable();
                                let food = costs.food.as_human_readable();

                                if money == materials && materials == food {
                                    // all resource metrics are the same
                                    // this is currently the only used metrics growth
                                    ui.horizontal(|ui| {
                                        let color = if &*metrics >= costs {
                                            REDRAW_AFFORD_COLOR
                                        } else {
                                            REDRAW_COLOR
                                        };
                                        ui.add(
                                            egui::widgets::Image::new(SizedTexture::new(
                                                resources_icon,
                                                egui::Vec2::new(
                                                    button_size * 0.4,
                                                    button_size * 0.4,
                                                ),
                                            ))
                                            .tint(color),
                                        );
                                        ui.label(
                                            RichText::new(money)
                                                .font(FontId::new(
                                                    window.height() * SMALL_TEXT_SIZE / 1.4,
                                                    egui::FontFamily::Name(BOLD_FONT_NAME.into()),
                                                ))
                                                .color(color),
                                        );
                                    });
                                } else {
                                    // not used in any game mode
                                    // this is basically a placeholder
                                    unimplemented!();
                                }

                                // time remaining
                                ui.horizontal(|ui| {
                                    ui.add(
                                        egui::widgets::Image::new(SizedTexture::new(
                                            hourglass_icon,
                                            egui::Vec2::new(button_size * 0.3, button_size * 0.3),
                                        ))
                                        .tint(BUTTON_COLOR),
                                    );
                                    ui.label(
                                        RichText::new(format!(
                                            "{}",
                                            hand_interval.0.remaining_secs() as usize
                                        ))
                                        .font(FontId::new(
                                            window.height() * SMALL_TEXT_SIZE / 2.2,
                                            egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                        ))
                                        .color(BUTTON_COLOR),
                                    );
                                });
                            }
                        });
                });
            // check if a click happended this frame
            // and if it happended in the allocated area
            // XXX: required because the text area doesn't pass through clicks
            if let Some(click_pos) = ui.ctx().input(|i| {
                if i.pointer.any_click() {
                    i.pointer.interact_pos()
                } else {
                    None
                }
            }) {
                if resp.response.rect.contains(click_pos) {
                    // click happened inside the allocated area
                    // => recognise redraw press
                    redraw_writer.send(ManualRedrawEvent);
                }
            }
        });
}

fn select_slot_using_keybindings(
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    mut hand: ResMut<Hand>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }
    for (i, keybinding) in [
        &settings.keybindings.select_card_1,
        &settings.keybindings.select_card_2,
        &settings.keybindings.select_card_3,
        &settings.keybindings.select_card_4,
        &settings.keybindings.select_card_5,
    ]
    .iter()
    .enumerate()
    {
        if keybinding.any(|code| input.just_pressed(code)) {
            hand.selected = Some(i);
        }
    }

    // previous slot
    if settings
        .keybindings
        .select_card_prev
        .any(|code| input.just_pressed(code))
    {
        hand.select_previous_slot();
    }
    // next slot
    if settings
        .keybindings
        .select_card_next
        .any(|code| input.just_pressed(code))
    {
        hand.select_next_slot();
    }
}

fn redraw_keybinding(
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    mut ev_redraw: EventWriter<ManualRedrawEvent>,
) {
    if settings
        .0
        .keybindings
        .redraw_cards
        .any(|code| input.just_pressed(code))
    {
        ev_redraw.send(ManualRedrawEvent);
    }
}
