use bevy::prelude::*;
use bevy_egui::{
    egui::{self, FontId, RichText},
    EguiContexts,
};
use chrono::DateTime;

use crate::{
    define_asset_collection,
    game::{
        gamemodes::{challenge::ChallengeGameState, zen::ZenGameState},
        recorder::{recorder_activated, SaveRecord, SavedRecord},
    },
    i18n::Localization,
    prelude::*,
    settings::get_record_name,
    ui::{
        theme::{
            BIG_TEXT_SIZE, BUTTON_COLOR, BUTTON_TEXT_COLOR, NORMAL_TEXT_COLOR, OUTER_MARGIN,
            PARAGRAPH_FONT_NAME, POPUP_COLOR_BG, POPUP_RADIUS, POPUP_STROKE_COLOR,
            SECONDARY_BUTTON_SIZE, SMALL_TEXT_SIZE, SPACE_SIZE,
        },
        UiClickEvent,
    },
};

use super::{
    hand::render_redraw_button,
    layout::{init_layout_manager, LayoutManager},
    pause::PauseUiState,
    IconButtonCard,
};

const RECORD_NAME: &str = "tte-record";

macro_rules! define_game_export_plugin {
    ($({
        // state in which game_export should be setable
        state: $state:expr,
    }),*) => {
        pub struct GameExportUi;
        impl Plugin for GameExportUi {
            fn build(&self, app: &mut App) {
                app.register_asset_collection::<ShareIcon>();
                app.init_resource::<ExportWindow>();

                app.add_systems(
                    Update,
                    show_export_button
                        .run_if(recorder_activated)
                        .run_if(|state: Res<ExportWindow>| !state.open)
                        .run_if(never()$(.or(in_state($state)))*)
                        .after(init_layout_manager)
                );

                app.add_systems(
                    Update,
                    show_export_window
                        .run_if(recorder_activated)
                        .run_if(|state: Res<ExportWindow>| state.open)
                        .run_if(never()$(.or(in_state($state)))*)
                        .after(render_redraw_button)
                        .after(show_export_button)
                );
                $(
                    app.add_systems(
                        OnExit($state),
                        |mut state: ResMut<ExportWindow>| { *state = Default::default(); }
                    );
                )*
            }
        }
    };
}

define_game_export_plugin![
    {
        state: ChallengeGameState::GameOver,
    },
    {
        state: ChallengeGameState::Spectate,
    },
    {
        state: ChallengeGameState::BetweenRounds,
    },
    {
        state: PauseUiState::Shown,
    },
    {
        state: ZenGameState::BetweenRounds,
    }
];

define_asset_collection!(
    ShareIcon,
    !share : Image = "icons/buttons/share.png",
    err : "buttton-icon-missing" "Button icon missing." "button-icon-missing-desc" "You can not use a couple of buttons without the icons"
);

#[derive(Debug, Default, Clone)]
pub enum ExportState {
    #[default]
    None,
    Ok,
    Err,
}

#[derive(Resource, Default)]
pub struct ExportWindow {
    /// whether the export window is open
    pub open: bool,
    /// export state
    pub exported: ExportState,
    /// path where the file be saved to (cache)
    pub export_path: Option<String>,
}

pub fn show_export_button(
    windows: Query<&Window>,
    mut layout: ResMut<LayoutManager>,
    mut contexts: EguiContexts,
    mut click_event: EventWriter<UiClickEvent>,
    icon: Res<ShareIcon>,
    mut state: ResMut<ExportWindow>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.width() == 0.0 || window.height() == 0.0 {
        return;
    }

    let icon_size = window.width().min(window.height()) * BIG_TEXT_SIZE / 2.0;

    let icon_texture = contexts.add_image(icon.share.clone_weak());

    let icon = IconButtonCard {
        icon_size,
        texture: icon_texture,
    };

    let cell = if let Some(cell) = layout.allocate_margin(
        super::layout::LayoutOrigin::RightBottom,
        super::layout::LayoutSizeRequest::Fixed(icon.get_size()),
        super::layout::LayoutSizeRequest::Fixed(icon.get_size()),
        super::layout::LayoutDirection::Horizontal,
        OUTER_MARGIN,
    ) {
        cell
    } else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    egui::Area::new("export-button".into())
        .pivot(egui::Align2::RIGHT_TOP)
        .fixed_pos(cell.rect.right_top())
        .show(egui, |ui| {
            if ui.add(icon).clicked() {
                click_event.send(UiClickEvent);
                state.open = true;
                state.exported = Default::default();
                let local: DateTime<chrono::offset::Local> = chrono::offset::Local::now();
                state.export_path = get_record_name(
                    local
                        .format(&format!("{}-%Y-%m-%d_%H%M%S.json", RECORD_NAME))
                        .to_string(),
                );
            }
        });
}

pub fn show_export_window(
    mut contexts: EguiContexts,
    mut state: ResMut<ExportWindow>,
    localization: Res<Localization>,
    windows: Query<&Window>,
    mut click_event: EventWriter<UiClickEvent>,
    mut save_event: EventWriter<SaveRecord>,
    mut save_event_res: EventReader<SavedRecord>,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let window = windows.single();
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let my_frame = egui::Frame {
        fill: POPUP_COLOR_BG,
        stroke: egui::Stroke::new(1.5, POPUP_STROKE_COLOR),
        corner_radius: POPUP_RADIUS,
        inner_margin: egui::Margin::same(5),
        outer_margin: egui::Margin::same(0),
        ..Default::default()
    };

    for SavedRecord(succ) in save_event_res.read() {
        state.exported = if *succ {
            ExportState::Ok
        } else {
            ExportState::Err
        };
    }

    let screen_size = egui.input(|i| i.screen_rect);
    egui::Area::new("export-window-backdrop".into()).show(egui, |ui| {
        egui::Frame {
            fill: POPUP_COLOR_BG.gamma_multiply(0.7),
            ..Default::default()
        }
        .show(ui, |ui| {
            if ui
                .allocate_rect(screen_size, egui::Sense::click())
                .clicked()
            {
                state.open = false;
            }
        });
    });

    let mut open = state.open;

    egui::Window::new(localization.translate("export-window-title"))
        .frame(my_frame)
        .collapsible(false)
        .resizable(false)
        .open(&mut open)
        .anchor(egui::Align2::CENTER_CENTER, [0.0, 0.0])
        .order(egui::Order::Foreground)
        .show(egui, |ui| {
            match &mut state.export_path {
                Some(path) => {
                    ui.label(
                        RichText::new(format!(
                            "{}: ",
                            localization.translate("export-window-desc")
                        ))
                        .color(NORMAL_TEXT_COLOR),
                    );
                    ui.add(
                        egui::text_edit::TextEdit::singleline(path)
                            .min_size(egui::Vec2::new(window.x * 0.3, window.y * 0.02))
                            .font(FontId::new(
                                window.y * SMALL_TEXT_SIZE,
                                egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                            )),
                    );
                }
                None => {
                    ui.label(
                        RichText::new(format!(
                            "{}: ",
                            localization.translate("export-window-desc-err")
                        ))
                        .color(NORMAL_TEXT_COLOR),
                    );
                    let mut path = String::new();
                    ui.add(
                        egui::text_edit::TextEdit::singleline(&mut path)
                            .min_size(egui::Vec2::new(window.x * 0.2, window.y * 0.02))
                            .font(FontId::new(
                                window.y * SMALL_TEXT_SIZE,
                                egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                            )),
                    );
                    if !path.is_empty() {
                        state.export_path = Some(path);
                    }
                }
            }

            ui.add_space(window.y * SPACE_SIZE);

            match state.exported {
                ExportState::Ok => {
                    ui.label(
                        RichText::new(localization.translate("export-window-succ"))
                            .color(NORMAL_TEXT_COLOR),
                    );
                }
                ExportState::Err => {
                    ui.label(
                        RichText::new(localization.translate("export-window-err"))
                            .color(NORMAL_TEXT_COLOR),
                    );
                }
                _ => (),
            };
            ui.add_space(window.y * SPACE_SIZE);

            if ui
                .add_sized(
                    (window * SECONDARY_BUTTON_SIZE).cast(),
                    egui::Button::new(
                        RichText::new(localization.translate("export-window-save-game"))
                            .color(BUTTON_TEXT_COLOR),
                    )
                    .fill(BUTTON_COLOR),
                )
                .clicked()
            {
                if let Some(path) = &state.export_path {
                    save_event.send(SaveRecord(path.clone()));
                    click_event.send(UiClickEvent);
                }
            }
        });

    state.open = open;
}
