use bevy::prelude::*;
use bevy_egui::{
    egui::{self, Align2, Area, InnerResponse, ScrollArea, Ui},
    EguiContexts,
};

use crate::{
    game::{
        controls::creative_placer::CreativeInventory,
        gamemodes::{creative::CreativeGameState, GameMode},
        tiles::{assets::TilePreviews, TileType},
        GamePauseState,
    },
    i18n::Localization,
    prelude::Translate,
    settings::ActiveSettingsBank,
    ui::{
        commons::ShowGameUi,
        is_typing_egui,
        theme::{
            BACKGROUND_COLOR, CREATIVE_INV_SIZE_LANDSCAPE, CREATIVE_INV_SIZE_PORTRAIT,
            CREATIVE_INV_SLOT_COUNT_LANDSCAPE, CREATIVE_INV_SLOT_COUNT_PORTRAIT, INV_SLOT_SIZE,
            INV_SLOT_STROKE, OUTER_MARGIN,
        },
    },
};

use super::{
    inventory::InventorySlot, layout::LayoutManager, pause::ui_pause_button, CommonInventoryAssets,
    WithTranslationID,
};

macro_rules! define_creative_inventory_ui_plugin {
    ($({
        // gamemode in which the inventory should be available
        gamemode: $gm:expr,
        // state in which to display the quick access slots
        // omit to render the slots yourself
        $(display_state: $display_state:expr,)?
        // number of slots in inventory
        inventory_size: $inv_size:expr$(,)?
    }),*) => {
        pub struct CreativeInventoryUiPlugin;
        impl Plugin for CreativeInventoryUiPlugin {
            fn build(&self, app: &mut App) {
                app.add_sub_state::<CreativeInventoryUiState>();
                app.insert_resource(SelectedCreativeInventorySlot(None));

                use crate::prelude::never;

                app.add_systems(
                    Update,
                    control_creative_inv
                        .run_if(never()$(.or(in_state($gm)))*)
                        .run_if(in_state(CreativeInventoryUiState::Opened)),
                );

                app.add_systems(
                    Update,
                    display_inventory_ui
                        .run_if(never()$(.or(in_state($gm)))*)
                        .run_if(in_state(CreativeInventoryUiState::Opened)),
                );
                app.add_systems(
                    Update,
                    open_inv_keybindings
                        .run_if(never()$(.or(in_state($gm)))*)
                        .run_if(in_state(GamePauseState::Running)),
                );
                app.add_systems(
                    Update,
                    close_inv_keybindings
                        .run_if(never()$(.or(in_state($gm)))*)
                        .run_if(in_state(CreativeInventoryUiState::Opened)),
                );
                // render quickaccess slots if the display_state is set
                app.add_systems(
                    Update,
                    ingame_inv_slots
                        .run_if(never()$($(.or(in_state($display_state)))?)*)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(in_state(ShowGameUi::On))
                        .after(ui_pause_button),
                );
                app.add_systems(
                    Update,
                    control_creative_inv_slots
                        .run_if(never()$(.or(in_state($gm)))*)
                        // by default we only control the inventory when it is opened
                        .run_if(in_state(CreativeInventoryUiState::Opened)
                                // this plugin also renders the quick access slots for these states
                                // so we have to control the inventory in the opened and closed state
                                .or(never()$($(.or(in_state($display_state)))?)*)),
                );
            }
        }
    }
}
#[cfg(not(debug_assertions))]
define_creative_inventory_ui_plugin![
    {
        gamemode: GameMode::Creative,
        display_state: CreativeGameState::InGame,
        inventory_size: CREATIVE_INV_SLOT_COUNT
    }
];
#[cfg(debug_assertions)]
define_creative_inventory_ui_plugin![
    {
        gamemode: GameMode::Creative,
        display_state: CreativeGameState::InGame,
        inventory_size: CREATIVE_INV_SLOT_COUNT
    },
    {
        gamemode: GameMode::Challenge,
        inventory_size: DEBUG_PLACER_SIZE
    }
];

#[derive(SubStates, Default, Debug, Hash, PartialEq, Eq, Clone)]
#[source(GamePauseState = GamePauseState::Paused)]
pub enum CreativeInventoryUiState {
    /// show the inventory ui
    Opened,
    /// do not show the inventory ui
    /// only the slots and open button
    #[default]
    Closed,
}

#[derive(Resource, Deref, DerefMut)]
/// Creative inventory slot focused
/// set to None if no keyboard input was registered
struct SelectedCreativeInventorySlot(Option<usize>);

fn display_inventory_ui(
    mut contexts: EguiContexts,
    mut inv: ResMut<CreativeInventory>,
    previews: Res<TilePreviews>,
    localization: Res<Localization>,
    handles: Res<CommonInventoryAssets>,
    mut pause_next: ResMut<NextState<GamePauseState>>,
    mut keyboard_selection: ResMut<SelectedCreativeInventorySlot>,
) {
    let egui_open_ico = contexts.add_image(handles.menu_open.clone_weak());

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let my_frame = egui::Frame {
        fill: BACKGROUND_COLOR.gamma_multiply(0.7),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };
    // center panel to fill the background
    egui::CentralPanel::default()
        .frame(my_frame)
        .show(egui, |_ui| {});

    let screen_size = egui.input(|i| i.screen_rect);

    let smallest_edge = screen_size.width().min(screen_size.height());
    // size required to fit the quick access bar
    let max_slot_size = smallest_edge * 0.8 / (inv.slots.len() + 1) as f32;
    let (col_count, width_perc) = if screen_size.width() > screen_size.height() {
        (
            CREATIVE_INV_SLOT_COUNT_LANDSCAPE,
            CREATIVE_INV_SIZE_LANDSCAPE,
        )
    } else {
        (CREATIVE_INV_SLOT_COUNT_PORTRAIT, CREATIVE_INV_SIZE_PORTRAIT)
    };
    let qa_slot_size = (smallest_edge * INV_SLOT_SIZE).min(max_slot_size);
    let slot_size = smallest_edge * width_perc / col_count as f32;

    let inventory = Area::new("creative-inventory-content".into())
        .anchor(Align2::CENTER_CENTER, [0.0, 0.0])
        .show(egui, |ui| {
            let tile_list = TileType::list();
            let mut row_count = tile_list.len() / col_count;
            if tile_list.len() % col_count != 0 {
                row_count += 1;
            }

            ScrollArea::vertical().show(ui, |ui| {
                for r in 0..row_count {
                    ui.horizontal(|ui| {
                        for c in 0..col_count {
                            let list_index = col_count * r + c;
                            if let Some(tile) = tile_list.get(list_index) {
                                if ui
                                    .add(InventorySlot {
                                        id: format!("inventory_grid_{}", tile.get_translation_id()),
                                        show_id: false,
                                        item: Some((
                                            previews
                                                .get_egui_handle(*tile)
                                                .expect("Every tile has a preview"),
                                            localization.translate(tile.get_translation_id()),
                                        )),
                                        size: slot_size,
                                        // show selected outline, if keyboard selector is active
                                        selected: keyboard_selection
                                            .map(|id| id == list_index)
                                            .unwrap_or(false),
                                        ..Default::default()
                                    })
                                    .clicked()
                                {
                                    // reset the keyboard selection as it uses a different mechanism
                                    // and cannot be used in connection with the touch/mouse workflow
                                    **keyboard_selection = None;
                                    // swap tiles
                                    let index = inv.selected;
                                    if inv.slots[index] == Some(*tile) {
                                        // simulated item put-back
                                        inv.slots[index] = None;
                                    } else {
                                        // the item isn't already in the slot
                                        // so put it in there
                                        inv.slots[index] = Some(*tile);
                                    }
                                }
                            }
                        }
                    });
                }
            });
        });

    let slot_size = qa_slot_size;
    Area::new("creative-inventory-slots".into())
        .pivot(egui::Align2::CENTER_TOP)
        //// make space twice as wide as distance between slots
        .fixed_pos(inventory.response.rect.center_bottom() + egui::Vec2::new(0.0, slot_size / 2.0))
        .show(egui, |ui| {
            let slot_count = inv.slots.len();
            let initial_size = egui::Vec2::new(slot_count as f32 * slot_size, slot_size);
            // manually creative a horizontal layout with fixed with
            // because otherwise it would take up the full width
            // and not be centered
            ui.allocate_ui_with_layout(
                initial_size,
                egui::Layout::left_to_right(egui::Align::Center),
                |ui| {
                    // render slots
                    for (index, slot) in inv.slots.clone().iter().enumerate() {
                        if ui
                            .add(InventorySlot {
                                id: (index + 1).to_string(),
                                show_id: true,
                                selected: index == inv.selected,
                                size: slot_size,
                                item: slot.map(|tile| {
                                    (
                                        previews
                                            .get_egui_handle(tile)
                                            .expect("Every tile has a preview"),
                                        localization.translate(tile.get_translation_id()),
                                    )
                                }),
                                ..Default::default()
                            })
                            .clicked()
                        {
                            inv.selected = index;
                        }
                    }
                    // render close button
                    if ui
                        .add(InventorySlot {
                            id: "inventory-close-button".to_string(),
                            show_id: false,
                            selected: false,
                            size: qa_slot_size,
                            item: Some((egui_open_ico, localization.translate("inventory-close"))),
                            tint: INV_SLOT_STROKE,
                        })
                        .clicked()
                    {
                        pause_next.set(GamePauseState::Running);
                    }
                },
            );
        });
}

fn open_inv_keybindings(
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    mut inv_state_next: ResMut<NextState<CreativeInventoryUiState>>,
    mut pause_next: ResMut<NextState<GamePauseState>>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }
    if settings
        .keybindings
        .inventory
        .any(|code| input.just_pressed(code))
    {
        pause_next.set(GamePauseState::Paused);
        inv_state_next.set(CreativeInventoryUiState::Opened);
    }
}

fn close_inv_keybindings(
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    mut pause_next: ResMut<NextState<GamePauseState>>,
) {
    // close inventory
    if settings
        .keybindings
        .inventory
        .any(|code| input.just_pressed(code))
        || settings
            .keybindings
            .menu
            .any(|code| input.just_pressed(code))
    {
        // automatically closes the inventory ui
        // either using the inventory keybinding
        // or the pause keybinding
        pause_next.set(GamePauseState::Running);
    }
}

/// controls the creative inventory quick access slots
/// this system most likely runs when the inventory is open and when it is closed
fn control_creative_inv_slots(
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    mut inv: ResMut<CreativeInventory>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }
    for (i, keybinding) in [
        &settings.keybindings.select_card_1,
        &settings.keybindings.select_card_2,
        &settings.keybindings.select_card_3,
        &settings.keybindings.select_card_4,
        &settings.keybindings.select_card_5,
    ]
    .iter()
    .take(inv.slots.len().min(5))
    .enumerate()
    {
        if keybinding.any(|code| input.just_pressed(code)) {
            inv.selected = i;
        }
    }
    // previous slot
    if settings
        .keybindings
        .select_card_prev
        .any(|code| input.just_pressed(code))
    {
        if inv.selected == 0 {
            inv.selected = inv.slots.len() - 1;
        } else {
            inv.selected -= 1;
        }
    }
    // next slot
    if settings
        .keybindings
        .select_card_next
        .any(|code| input.just_pressed(code))
    {
        if inv.selected + 1 >= inv.slots.len() {
            inv.selected = 0;
        } else {
            inv.selected += 1;
        }
    }
}

/// controls the focus of the creative inventory tile list
/// only runs when the creative inventory is opened
fn control_creative_inv(
    mut contexts: EguiContexts,
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    mut inv: ResMut<CreativeInventory>,
    mut keyboard_selection: ResMut<SelectedCreativeInventorySlot>,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };
    let screen_size = egui.input(|i| i.screen_rect);

    let col_count = if screen_size.width() > screen_size.height() {
        CREATIVE_INV_SLOT_COUNT_LANDSCAPE
    } else {
        CREATIVE_INV_SLOT_COUNT_PORTRAIT
    };

    let tile_list = TileType::list();
    let mut row_count = tile_list.len() / col_count;
    if tile_list.len() % col_count != 0 {
        row_count += 1;
    }

    // correct selection errors
    // (should never occur, as the tile list stays the same)
    if let Some(list_index) = &mut **keyboard_selection {
        // keyboard selector is active
        if tile_list.len() <= *list_index {
            // Assumes that the tile list is never empty
            *list_index = 0;
        }
    }

    // x/y focus movement
    {
        let mut dr: isize = 0;
        let mut dc: isize = 0;

        // read keyboard input and change delta values
        if settings
            .keybindings
            .inventory_focus_up
            .any(|code| input.just_pressed(code))
        {
            dr -= 1;
        }
        if settings
            .keybindings
            .inventory_focus_down
            .any(|code| input.just_pressed(code))
        {
            dr += 1;
        }
        if settings
            .keybindings
            .inventory_focus_left
            .any(|code| input.just_pressed(code))
        {
            dc -= 1;
        }
        if settings
            .keybindings
            .inventory_focus_right
            .any(|code| input.just_pressed(code))
        {
            dc += 1;
        }

        if dr != 0 || dc != 0 {
            if let Some(list_index) = &mut **keyboard_selection {
                // modify existing selection
                let current_row = *list_index / col_count;
                let current_col = *list_index % col_count;
                // the inventory grid is not filled perfectly
                // which means that the last row might not contain col_count tiles
                // keybindings that wrap around thus have less columns to work with
                let cols_at_row = if current_row == row_count - 1 {
                    tile_list.len() % col_count
                } else {
                    col_count
                };
                let c = if dc >= 0 {
                    let a = current_col + dc as usize;
                    if a >= cols_at_row {
                        0
                    } else {
                        a
                    }
                } else if dc.unsigned_abs() > current_col {
                    cols_at_row - 1
                } else {
                    current_col - dc.unsigned_abs()
                };
                // the inventory grid is not filled perfectly
                // which means that the last row might not contain col_count tiles
                // keybindings that might wrap around thus have one row less to wrap
                let rows_at_col = if c >= tile_list.len() % col_count {
                    row_count - 1
                } else {
                    row_count
                };
                let r = if dr >= 0 {
                    let a = current_row + dr as usize;

                    if a >= rows_at_col {
                        0
                    } else {
                        a
                    }
                } else if dr.unsigned_abs() > current_row {
                    rows_at_col - 1
                } else {
                    current_row - dr.unsigned_abs()
                };
                let list_index = col_count * r + c;
                **keyboard_selection = Some(list_index)
            } else {
                // keyboard selector was only now toggled
                // always start on top left
                **keyboard_selection = Some(0);
            }
        }
    }
    // take gesture
    if settings
        .keybindings
        .inventory_take
        .any(|code| input.just_pressed(code))
    {
        if let Some(list_index) = **keyboard_selection {
            // keyboard selector is active
            if let Some(tile) = tile_list.get(list_index) {
                // and focused on a tile focused on a tile - execute swap action
                let index = inv.selected;
                if inv.slots[index] == Some(*tile) {
                    // simulated item put-back
                    inv.slots[index] = None;
                } else {
                    // the item isn't already in the slot
                    // so put it in there
                    inv.slots[index] = Some(*tile);
                }
            }
        } else {
            // keyboard selector was inactive
            // focus on first slot, as it should be triggert by keyboard input
            **keyboard_selection = Some(0);
        }
    }
}

#[allow(clippy::too_many_arguments)]
pub fn ingame_inv_slots(
    mut contexts: EguiContexts,
    mut inv: ResMut<CreativeInventory>,
    mut layout_mgr: ResMut<LayoutManager>,
    previews: Res<TilePreviews>,
    localization: Res<Localization>,
    handles: Res<CommonInventoryAssets>,
    mut inv_next: ResMut<NextState<CreativeInventoryUiState>>,
    mut pause_next: ResMut<NextState<GamePauseState>>,
) {
    let egui_open_ico = contexts.add_image(handles.menu_open.clone_weak());

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let screen_size = egui.input(|i| i.screen_rect);

    let slot_size = screen_size.width().min(screen_size.height()) * INV_SLOT_SIZE;

    let is_landscape = screen_size.width() > screen_size.height();
    let area = if is_landscape {
        // landscape mode
        // display inventory at bottom
        let Some(cell) = layout_mgr.allocate_margin(
            super::layout::LayoutOrigin::LeftBottom,
            super::layout::LayoutSizeRequest::Filled,
            super::layout::LayoutSizeRequest::Fixed(slot_size),
            super::layout::LayoutDirection::Horizontal,
            OUTER_MARGIN,
        ) else {
            return;
        };
        egui::Area::new("inventory-slots-quickaccess-landscape".into())
            .pivot(egui::Align2::CENTER_BOTTOM)
            .fixed_pos(cell.rect.center_bottom())
    } else {
        // portrait mode
        // display inventory on the right
        let Some(cell) = layout_mgr.allocate_margin(
            super::layout::LayoutOrigin::RightBottom,
            super::layout::LayoutSizeRequest::Fixed(slot_size),
            super::layout::LayoutSizeRequest::Filled,
            super::layout::LayoutDirection::Vertical,
            OUTER_MARGIN,
        ) else {
            return;
        };
        egui::Area::new("inventory-slots-quickaccess-portrait".into())
            .pivot(egui::Align2::RIGHT_CENTER)
            .fixed_pos(cell.rect.right_center())
    };

    area.show(egui, |ui| {
        dynamic_layout(ui, screen_size, |ui| {
            // render item slots themselves
            for (index, slot) in inv.slots.clone().iter().enumerate() {
                if ui
                    .add(InventorySlot {
                        id: (index + 1).to_string(),
                        show_id: true,
                        selected: index == inv.selected,
                        size: slot_size,
                        item: slot.map(|tile| {
                            (
                                previews
                                    .get_egui_handle(tile)
                                    .expect("Every tile has a preview"),
                                localization.translate(tile.get_translation_id()),
                            )
                        }),
                        ..Default::default()
                    })
                    .clicked()
                {
                    inv.selected = index;
                }
            }
            // render open inventory button
            if ui
                .add(InventorySlot {
                    id: "inventory-open-button".to_string(),
                    show_id: false,
                    selected: false,
                    size: slot_size,
                    item: Some((egui_open_ico, localization.translate("inventory-open"))),
                    tint: INV_SLOT_STROKE,
                })
                .clicked()
            {
                pause_next.set(GamePauseState::Paused);
                inv_next.set(CreativeInventoryUiState::Opened);
            }
        });
    });
}

/// wraps the ui in a horizontal wrapper if the screen width is larger than the height
/// otherwise a vertical layout will be used
pub fn dynamic_layout<R>(
    ui: &mut Ui,
    screen_size: egui::Rect,
    add_contents: impl FnOnce(&mut Ui) -> R,
) -> InnerResponse<R> {
    let is_landscape = screen_size.width() > screen_size.height();
    if is_landscape {
        ui.horizontal(add_contents)
    } else {
        ui.vertical(add_contents)
    }
}
