// use std::io::Read;

use std::marker::PhantomData;

use crate::{
    errors::ErrorDisplay,
    game::asset_loading::{
        register_and_load_assets, reset_asset_tracker, wait_for_assets, AppLoadingState,
        AssetLoadStateDump, AssetLoadedEvent, LoadableAssetCollection,
    },
    i18n::{Localization, Translate},
    AppState, DisplayMode,
};
use bevy::{asset::AssetLoader, prelude::*};
use bevy_egui::{
    egui::{FontData, FontDefinitions, FontFamily},
    EguiContexts,
};
use thiserror::{self, Error};

use super::theme::{BOLD_FONT_NAME, MONOSPACE_FONT_NAME, PARAGRAPH_FONT_NAME, TITLE_FONT_NAME};

#[derive(Asset, TypePath)]
pub struct FontResourceAsset(pub Vec<u8>);

#[derive(Debug, Error)]
#[non_exhaustive]
pub enum FontResourceLoadingError {
    #[error("Could not load font: {0}")]
    Io(#[from] std::io::Error),
}

#[derive(Default)]
pub struct FontResourceLoader;
impl AssetLoader for FontResourceLoader {
    type Asset = FontResourceAsset;
    type Settings = ();
    type Error = FontResourceLoadingError;

    async fn load(
        &self,
        reader: &mut dyn bevy::asset::io::Reader,
        _settings: &(),
        _load_context: &mut bevy::asset::LoadContext<'_>,
    ) -> Result<Self::Asset, Self::Error> {
        let mut bytes = Vec::new();
        reader.read_to_end(&mut bytes).await?;
        Ok(FontResourceAsset(bytes))
    }
}

#[derive(Resource)]
pub struct AppFonts {
    // TODO: register font attributes here
    permanent_marker: Handle<FontResourceAsset>,
    inter: Handle<FontResourceAsset>,
    inter_bold: Handle<FontResourceAsset>,
    fira_mono: Handle<FontResourceAsset>,
}
impl AppFonts {
    /// returns a list of available fonts
    fn list(&self) -> [(&str, Handle<FontResourceAsset>); 4] {
        [
            // TODO: add new font names here
            ("Permanent Marker", self.permanent_marker.clone_weak()),
            ("Inter", self.inter.clone_weak()),
            ("Inter (Bold)", self.inter_bold.clone_weak()),
            ("Fira Mono", self.fira_mono.clone_weak()),
        ]
    }
    /// returns a list of egui fontfamilies
    /// and a list of the fonts they consist of
    fn lookup(&self) -> [(&str, Vec<&str>); 4] {
        [
            // TODO: add new egui fonts here
            // syntax (font_name, [list, of, fallback, fonts])
            (TITLE_FONT_NAME, vec!["Permanent Marker", "Inter"]),
            (PARAGRAPH_FONT_NAME, vec!["Inter"]),
            (BOLD_FONT_NAME, vec!["Inter (Bold)"]),
            (MONOSPACE_FONT_NAME, vec!["Fira Mono"]),
        ]
    }
}
impl LoadableAssetCollection for AppFonts {
    /// initializes the resource
    /// loads all required assets directly into a handle,
    /// optional assets are wrappen in Options
    fn load_all(asset_server: &AssetServer) -> Self {
        Self {
            // TODO: add font asset paths here
            permanent_marker: asset_server
                .load("fonts/PermanentMarker/PermanentMarker-Regular.ttf"),
            inter: asset_server.load("fonts/Inter/Inter-Regular.ttf"),
            inter_bold: asset_server.load("fonts/Inter/Inter-Bold.ttf"),
            fira_mono: asset_server.load("fonts/Fira/FiraMono-Regular.ttf"),
        }
    }
    /// checks if all handles were loaded
    /// if optional handles failed to load, they will be reset to None
    fn check_all(&mut self, asset_server: &AssetServer) -> AssetLoadStateDump {
        let mut dump = AssetLoadStateDump::default();
        // required later, once the assets were loaded
        dump.requested += self.list().len();
        for (_, handle) in self.list() {
            dump.requested += 1;
            match asset_server.get_load_state(&handle) {
                Some(bevy::asset::LoadState::Loaded) => dump.loaded += 1,
                Some(bevy::asset::LoadState::Failed(_)) => dump.failed += 1,
                _ => (),
            }
        }
        dump
    }

    /// return the error display configuration
    /// return None to ignore the error
    #[allow(unused_variables)]
    fn get_error(&self, localization: Option<&Localization>) -> Option<ErrorDisplay> {
        Some(ErrorDisplay {
            title: localization
                .and_then(|l| l.try_translate("missing-font-title"))
                .unwrap_or(String::from("Not all fonts could be loaded")),
            description: localization
                .and_then(|l| l.try_translate("missing-font-description"))
                .unwrap_or(String::from(
                    "The game would look pretty boring without custom fonts.",
                )),
            link: None,
        })
    }

    /// registers all required components to load the collection
    /// you probably want to use `LACManager.register_asset_collection` instead
    fn register(app: &mut App) -> &mut App {
        app.add_systems(
            Startup,
            register_and_load_assets::<Self>.run_if(in_state(DisplayMode::Graphic)),
        );
        app.add_event::<AssetLoadedEvent<Self>>();
        app.add_systems(
            Update,
            check_asset_load_state_with_font
                .run_if(in_state(AppState::LoadApp))
                .run_if(in_state(DisplayMode::Graphic))
                .before(wait_for_assets)
                .after(reset_asset_tracker),
        );
        app
    }
}

#[allow(clippy::too_many_arguments)]
fn check_asset_load_state_with_font(
    mut collection: ResMut<AppFonts>,
    mut state: ResMut<AppLoadingState>,
    asset_server: Res<AssetServer>,
    mut next: ResMut<NextState<AppState>>,
    mut commands: Commands,
    localization: Option<Res<Localization>>,
    mut event: EventWriter<AssetLoadedEvent<AppFonts>>,
    mut finished_state: Local<Option<AssetLoadStateDump>>,
    assets: Res<Assets<FontResourceAsset>>,
    mut contexts: EguiContexts,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    // the asset finished loading
    if let Some(dump) = &*finished_state {
        state.loaded += dump.loaded;
        state.failed += dump.failed;
        state.ignored += dump.ignored;
        return;
    }

    let mut dump = collection.check_all(asset_server.into_inner());

    if dump.failed > 0 {
        // show the error screen if the collection has an error message
        if let Some(msg) = collection.get_error(localization.as_deref()) {
            next.set(AppState::Error);
            commands.spawn(msg);
        }
    }

    // the handles were loaded successfully
    if dump.requested == dump.loaded + collection.list().len() {
        // register font with egui
        let mut font_definitions = FontDefinitions::default();
        for (name, handle) in collection.list() {
            let Some(font) = assets.get(&handle) else {
                error!("Failed to get font data");
                dump.failed += 1;
                continue;
            };
            font_definitions.font_data.insert(
                name.to_string(),
                FontData::from_owned(font.0.to_owned()).into(),
            );
            dump.loaded += 1;
        }
        // register font family
        let system_fonts = FontDefinitions::builtin_font_names();
        for (name, list) in collection.lookup() {
            let mut inner_fonts = list.iter().map(|s| s.to_string()).collect::<Vec<String>>();
            inner_fonts.append(
                &mut system_fonts
                    .iter()
                    .map(|s| s.to_string())
                    .collect::<Vec<String>>(),
            );
            font_definitions
                .families
                .insert(FontFamily::Name(name.into()), inner_fonts);
        }
        // set font definitons
        egui.set_fonts(font_definitions);
    }

    state.loaded += dump.loaded;
    state.failed += dump.failed;
    state.ignored += dump.ignored;

    if dump.requested == dump.loaded {
        // mark the collection as loaded
        *finished_state = Some(dump);
        // notifiy possible listeners
        event.send(AssetLoadedEvent(PhantomData {}));
    }
}

pub fn set_fallback_fonts(mut contexts: EguiContexts) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };
    let system_fonts = FontDefinitions::builtin_font_names();
    let mut font_definitions = FontDefinitions::default();
    // register font family
    for name in [BOLD_FONT_NAME, TITLE_FONT_NAME, PARAGRAPH_FONT_NAME] {
        font_definitions.families.insert(
            FontFamily::Name(name.into()),
            system_fonts
                .iter()
                .map(|s| s.to_string())
                .collect::<Vec<String>>(),
        );
    }
    // set font definitons
    egui.set_fonts(font_definitions);
}
