use bevy::{asset::LoadState, prelude::*, time::Timer, utils::Duration};
use bevy_egui::{
    egui::{self, load::SizedTexture, Color32},
    EguiContexts,
};

use crate::{
    define_asset_collection,
    game::asset_loading::{
        wait_for_assets, AppLoadingDelay, AppLoadingState, AssetLoadedEvent, LACManager,
    },
    AppState,
};

use super::theme::{
    BACKGROUND_COLOR, METRICS_BAR_BG_COLOR, METRICS_BAR_FG_COLOR, METRICS_BAR_TEXT_COLOR,
    OUTER_MARGIN,
};

/// height in percent of smallest display side
const BANNER_HEIGHT: f32 = 0.4;
/// aspect-ratio of the banner image
/// width over height
const BANNER_RATIO: f32 = 1920. / 1080.;
const BANNER_FADE_DURATION: Duration = Duration::from_millis(500);
/// width in percent of banner width
const BAR_WIDTH: f32 = 0.5;

pub struct LoadingScreenPlugin;
impl Plugin for LoadingScreenPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<LoadingScreenAssets>();
        app.init_resource::<BannerFadeState>();
        app.add_systems(
            Update,
            show_loading_screen
                .run_if(in_state(AppState::LoadApp))
                .after(wait_for_assets),
        );
        app.add_systems(
            Update,
            (step_fade_timer, start_fade_on_load).run_if(in_state(AppState::LoadApp)),
        );
    }
}

#[derive(Resource, Deref, DerefMut)]
pub struct BannerFadeState(Timer);
impl Default for BannerFadeState {
    fn default() -> Self {
        let mut t = Timer::new(BANNER_FADE_DURATION, TimerMode::Once);
        // pause the timer by default
        // will be unpaused once the banner finished loading
        t.pause();
        Self(t)
    }
}
fn step_fade_timer(mut fade: ResMut<BannerFadeState>, time: Res<Time>) {
    fade.tick(time.delta());
}
fn start_fade_on_load(
    mut events: EventReader<AssetLoadedEvent<LoadingScreenAssets>>,
    mut fade: ResMut<BannerFadeState>,
) {
    // nothing to do
    if events.is_empty() {
        return;
    }
    // start the fade timer
    fade.unpause();
    // clear tasks
    events.clear();
}

define_asset_collection!(
    LoadingScreenAssets,
    ?banner : Image = "banner.png",
);

pub fn show_loading_screen(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    delay: Res<AppLoadingDelay>,
    state: Res<AppLoadingState>,
    banner: Res<LoadingScreenAssets>,
    banner_fade: Res<BannerFadeState>,
    asset_server: Res<AssetServer>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    // we are waiting for all assets and the startup delay
    let needs = state.requested;
    let has = state.ignored + state.loaded;

    let banner_height = BANNER_HEIGHT
        * if window.x < window.y {
            window.x
        } else {
            window.y
        };
    let banner_width = BANNER_RATIO * banner_height;

    // register the banner texture with egui if available
    let banner_texture = if let Some(h) = &banner.banner {
        match asset_server.get_load_state(h) {
            Some(LoadState::Loaded) => Some(contexts.add_image(h.clone_weak())),
            _ => None,
        }
    } else {
        None
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let my_frame = egui::Frame {
        fill: BACKGROUND_COLOR,
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };

    // use central panel to fill the screen & paint background color
    egui::CentralPanel::default()
        .frame(my_frame)
        .show(egui, |_ui| {});

    egui::Area::new("loading-screen-bar".into())
        .anchor(egui::Align2::CENTER_CENTER, [0.0, 0.0])
        .show(egui, |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().interaction.selectable_labels = false;
                ui.style_mut().visuals.extreme_bg_color = METRICS_BAR_BG_COLOR;
                ui.style_mut().visuals.selection.stroke.color = METRICS_BAR_TEXT_COLOR;

                if let Some(banner_texture) = banner_texture {
                    // show banner image
                    ui.add(
                        egui::widgets::Image::new(SizedTexture::new(
                            banner_texture,
                            [banner_width, banner_height],
                        ))
                        .tint(Color32::WHITE.linear_multiply(banner_fade.fraction())),
                    );
                } else {
                    // add spacing as a placeholder for the banner
                    ui.add_space(banner_height);
                }

                ui.add_space(OUTER_MARGIN);

                let bar_width = BAR_WIDTH * banner_width;
                ui.add(
                    egui::ProgressBar::new((has as f32 / needs as f32) * delay.fraction())
                        .desired_width(bar_width)
                        .fill(METRICS_BAR_FG_COLOR),
                );
            });
        });
}
