use bevy::log::Level;
use serde::{de::Visitor, Deserialize, Serialize, Serializer};

use crate::{i18n::Localization, prelude::*};

#[derive(Serialize, Deserialize, Default, Debug, Clone)]
#[serde(default)]
pub struct LogSettings {
    #[serde(default)]
    pub level: LogLevelWrapper,
}
#[derive(Debug, PartialEq, Clone)]
pub enum LogLevelWrapper {
    /// disable logging
    Disabled,
    /// wrap bevy log level
    Level(Level),
}
impl Default for LogLevelWrapper {
    fn default() -> Self {
        Self::Level(Level::INFO)
    }
}
#[allow(dead_code)]
impl LogLevelWrapper {
    pub fn list() -> Vec<Self> {
        vec![
            Self::Disabled,
            Self::Level(Level::ERROR),
            Self::Level(Level::WARN),
            Self::Level(Level::INFO),
            Self::Level(Level::DEBUG),
            Self::Level(Level::TRACE),
        ]
    }
    pub fn get_name(&self, localization: &Localization) -> String {
        match self {
            LogLevelWrapper::Disabled => localization.translate("disabled").to_string(),
            LogLevelWrapper::Level(lv) => match *lv {
                Level::ERROR => localization.translate("error").to_string(),
                Level::WARN => localization.translate("warning").to_string(),
                Level::INFO => localization.translate("info").to_string(),
                Level::DEBUG => localization.translate("debug").to_string(),
                Level::TRACE => localization.translate("trace").to_string(),
            },
        }
    }
}
impl Serialize for LogLevelWrapper {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        match self {
            LogLevelWrapper::Disabled => serializer.serialize_str("off"),
            LogLevelWrapper::Level(lv) => match *lv {
                Level::ERROR => serializer.serialize_str("error"),
                Level::WARN => serializer.serialize_str("warn"),
                Level::INFO => serializer.serialize_str("info"),
                Level::DEBUG => serializer.serialize_str("debug"),
                Level::TRACE => serializer.serialize_str("trace"),
            },
        }
    }
}
struct LogLevelVisitor;
impl Visitor<'_> for LogLevelVisitor {
    type Value = LogLevelWrapper;

    fn expecting(&self, formatter: &mut std::fmt::Formatter) -> std::fmt::Result {
        formatter.write_str("a string containing one of: off, error, warn, info, debug, trace")
    }

    fn visit_str<E>(self, v: &str) -> Result<Self::Value, E>
    where
        E: serde::de::Error,
    {
        match v {
            "off" => Ok(LogLevelWrapper::Disabled),
            "error" => Ok(LogLevelWrapper::Level(Level::ERROR)),
            "warn" => Ok(LogLevelWrapper::Level(Level::WARN)),
            "info" => Ok(LogLevelWrapper::Level(Level::INFO)),
            "debug" => Ok(LogLevelWrapper::Level(Level::DEBUG)),
            "trace" => Ok(LogLevelWrapper::Level(Level::TRACE)),
            _ => Err(E::custom(format!("unknown log level: {}", v))),
        }
    }
    fn visit_string<E>(self, v: String) -> Result<Self::Value, E>
    where
        E: serde::de::Error,
    {
        self.visit_str(&v)
    }
}
impl<'de> Deserialize<'de> for LogLevelWrapper {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_string(LogLevelVisitor)
    }
}
