use serde::{Deserialize, Serialize};

use crate::{i18n::Localization, prelude::Translate};

#[derive(Serialize, Deserialize, Debug, Clone)]
#[serde(default)]
pub struct GraphicsSettings {
    pub vsync: bool,
    pub shadows: ShadowQuality,
    pub hdr: bool,
    pub debug_text: bool,
    pub animations: bool,
    pub water: bool,
}

impl Default for GraphicsSettings {
    fn default() -> Self {
        Self {
            vsync: true,
            shadows: ShadowQuality::Beautiful,
            hdr: false,
            debug_text: false,
            animations: true,
            water: true,
        }
    }
}

#[derive(Debug, Serialize, Deserialize, Clone, PartialEq)]
pub enum ShadowQuality {
    Max,
    Beautiful,
    Normal,
    Off,
}
impl ShadowQuality {
    /// returns a list of all ShadowQualities,
    pub fn list() -> Vec<Self> {
        vec![Self::Max, Self::Beautiful, Self::Normal, Self::Off]
    }
    pub fn get_name(&self, localization: &Localization) -> String {
        match self {
            ShadowQuality::Max => localization
                .translate("config-graphics-shadow-max")
                .to_string(),
            ShadowQuality::Beautiful => localization
                .translate("config-graphics-shadow-beautiful")
                .to_string(),
            ShadowQuality::Normal => localization
                .translate("config-graphics-shadow-normal")
                .to_string(),
            ShadowQuality::Off => localization
                .translate("config-graphics-shadow-off")
                .to_string(),
        }
    }
    pub fn enabled(&self) -> bool {
        match self {
            ShadowQuality::Max => true,
            ShadowQuality::Beautiful => true,
            ShadowQuality::Normal => true,
            ShadowQuality::Off => false,
        }
    }
    pub fn get_cascades(&self) -> usize {
        match self {
            ShadowQuality::Max => 4,
            ShadowQuality::Beautiful => 3,
            ShadowQuality::Normal => 2,
            ShadowQuality::Off => 1,
        }
    }
    pub fn get_max_distance(&self) -> f32 {
        match self {
            ShadowQuality::Max => 1000.,
            ShadowQuality::Beautiful => 500.,
            ShadowQuality::Normal => 100.,
            ShadowQuality::Off => 50.,
        }
    }
}
