use fastrand::Rng;

pub trait TTERand {
    fn choose_n<T: Iterator>(&mut self, source: T, amount: usize) -> Vec<T::Item>;
}

impl TTERand for Rng {
    fn choose_n<T: Iterator>(&mut self, mut source: T, amount: usize) -> Vec<T::Item> {
        // Adapted from: https://docs.rs/rand/latest/rand/seq/trait.IteratorRandom.html#method.choose_multiple
        // and from: https://github.com/smol-rs/fastrand/blob/master/src/lib.rs
        let mut reservoir = Vec::with_capacity(amount);

        reservoir.extend(source.by_ref().take(amount));

        // Continue unless the iterator was exhausted
        //
        // note: this prevents iterators that "restart" from causing problems.
        // If the iterator stops once, then so do we.
        if reservoir.len() == amount {
            for (i, elem) in source.enumerate() {
                let end = i + 1 + amount;
                let k = self.u32(0..end as u32) as usize;
                if let Some(slot) = reservoir.get_mut(k) {
                    *slot = elem;
                }
            }
        } else {
            // If less than one third of the `Vec` was used, reallocate
            // so that the unused space is not wasted. There is a corner
            // case where `amount` was much less than `self.len()`.
            if reservoir.capacity() > 3 * reservoir.len() {
                reservoir.shrink_to_fit();
            }
        }
        reservoir
    }
}
