use super::args::CliArgs;
use bevy::app::App;
use std::process::ExitCode;
use terratactician_expandoria::GameConfig;

#[cfg(target_family = "wasm")]
pub fn setup_replay(
    _app: &mut App,
    _args: &CliArgs,
    _config: &mut GameConfig,
) -> Result<bool, ExitCode> {
    Ok(false)
}

#[cfg(not(target_family = "wasm"))]
pub fn setup_replay(
    app: &mut App,
    args: &CliArgs,
    config: &mut GameConfig,
) -> Result<bool, ExitCode> {
    use bevy::{ecs::system::RunSystemOnce, prelude::*};
    use std::fs;
    use terratactician_expandoria::{
        game::{
            gamemodes::{campaign::CampaignModeConfig, GameMode},
            recorder::Record,
            replayer::{start_replay, Replay},
        },
        AppState,
    };

    if let Some(path) = &args.replay_file {
        if let Ok(content) = fs::read_to_string(path) {
            match serde_json::from_str::<Record>(&content) {
                Ok(replay) => {
                    if let Some(target_mode) = args.gamemode.get_mode() {
                        if target_mode != replay.mode {
                            eprintln!("Error while starting replay: The record contains a {:?} game, while a {:?} game was requested.", replay.mode, target_mode);
                            return Err(ExitCode::FAILURE);
                        }
                    }

                    // load nothing from settings
                    config.skip_recorder = true;
                    config.skip_seed = true;
                    config.skip_mode = true;

                    config.selected_gamemode = replay.mode.clone();
                    match replay.mode.clone() {
                        GameMode::Challenge => {
                            let Some(challenge_config) = replay.challenge.clone() else {
                                eprintln!("Error while parsing replay: missing challenge seed");
                                return Err(ExitCode::FAILURE);
                            };
                            app.insert_resource(challenge_config);
                        }
                        GameMode::Zen => {
                            let Some(zen_config) = replay.zen.clone() else {
                                eprintln!("Error while parsing replay: missing zen seed");
                                return Err(ExitCode::FAILURE);
                            };
                            app.insert_resource(zen_config);
                        }
                        GameMode::Creative => (),
                        GameMode::Campaign => {
                            let Some(wrapper) = replay.campaign.clone() else {
                                eprintln!("Error while parsing replay: missing campaign config");
                                return Err(ExitCode::FAILURE);
                            };
                            app.insert_resource(CampaignModeConfig { level: wrapper });
                        }
                    };

                    app.insert_resource(Replay(replay));
                    app.add_systems(
                        OnEnter(AppState::AppMenu),
                        (|mut commands: Commands| {
                            commands.queue(move |w: &mut World| {
                                w.run_system_once(start_replay).unwrap()
                            });
                        })
                        .run_if(run_once),
                    );
                }

                Err(e) => {
                    eprintln!("Error while parsing replay: {e}");
                    return Err(ExitCode::FAILURE);
                }
            }
        } else {
            eprintln!("Error while loading replay: file does not exists.");
            return Err(ExitCode::FAILURE);
        }
        Ok(true)
    } else {
        Ok(false)
    }
}
