use bevy::{app::App, ecs::system::RunSystemOnce, prelude::*};
use std::process::ExitCode;
#[cfg(target_family = "wasm")]
use terratactician_expandoria::quit;
use terratactician_expandoria::{
    game::{
        bot::{challenge_bot::start_challenge_bot_game, creative_bot::start_creative_bot_game},
        gamemodes::{
            campaign::{
                levels::{init_campaign_mode, CampaignGroupWrapper},
                CampaignGroupParserError, CampaignModeConfig,
            },
            challenge::init_challenge_mode,
            creative::init_creative_mode,
            zen::init_zen_mode,
            GameMode,
        },
    },
    AppState, GameConfig,
};

use super::args::CliArgs;

/// setups up game. Results in bool indicating, that a game was setup or an exit error.
pub fn setup_game(
    app: &mut App,
    args: &CliArgs,
    config: &mut GameConfig,
) -> Result<bool, ExitCode> {
    if args.gamemode.creative {
        setup_creative(app, args, config)?;
    } else if args.gamemode.challenge {
        setup_challenge(app, args, config)?;
    } else if args.gamemode.zen {
        setup_zen(app, args, config)?;
    } else if args.gamemode.campaign {
        setup_campaign(app, args, config)?;
    } else {
        return Ok(false);
    }
    Ok(true)
}

fn setup_creative(app: &mut App, args: &CliArgs, config: &mut GameConfig) -> Result<(), ExitCode> {
    config.selected_gamemode = GameMode::Creative;
    config.skip_mode = true;

    if args.bot.is_some() {
        app.add_systems(
            OnEnter(AppState::AppMenu),
            (|mut commands: Commands, mut app_state: ResMut<NextState<AppState>>| {
                app_state.set(AppState::SetupMenu);
                commands.queue(move |w: &mut World| {
                    w.run_system_once(start_creative_bot_game).unwrap()
                });
            })
            .run_if(run_once),
        );
    } else {
        app.add_systems(
            OnEnter(AppState::AppMenu),
            (|mut commands: Commands| {
                commands.queue(move |w: &mut World| w.run_system_once(init_creative_mode).unwrap());
            })
            .run_if(run_once),
        );
    }
    Ok(())
}
fn setup_challenge(app: &mut App, args: &CliArgs, config: &mut GameConfig) -> Result<(), ExitCode> {
    config.selected_gamemode = GameMode::Challenge;
    config.skip_mode = true;

    if args.bot.is_some() {
        app.add_systems(
            OnEnter(AppState::AppMenu),
            (|mut commands: Commands, mut app_state: ResMut<NextState<AppState>>| {
                app_state.set(AppState::SetupMenu);
                commands.queue(move |w: &mut World| {
                    w.run_system_once(start_challenge_bot_game).unwrap()
                });
            })
            .run_if(run_once),
        );
    } else {
        app.add_systems(
            OnEnter(AppState::AppMenu),
            (|mut commands: Commands| {
                commands
                    .queue(move |w: &mut World| w.run_system_once(init_challenge_mode).unwrap());
            })
            .run_if(run_once),
        );
    }
    Ok(())
}
fn setup_zen(app: &mut App, args: &CliArgs, config: &mut GameConfig) -> Result<(), ExitCode> {
    config.selected_gamemode = GameMode::Zen;
    config.skip_mode = true;

    if args.bot.is_some() {
        app.add_systems(
            OnEnter(AppState::AppMenu),
            (|mut commands: Commands, mut app_state: ResMut<NextState<AppState>>| {
                app_state.set(AppState::SetupMenu);
                commands.queue(move |w: &mut World| {
                    w.run_system_once(start_challenge_bot_game).unwrap()
                });
            })
            .run_if(run_once),
        );
    } else {
        app.add_systems(
            OnEnter(AppState::AppMenu),
            (|mut commands: Commands| {
                commands.queue(move |w: &mut World| w.run_system_once(init_zen_mode).unwrap());
            })
            .run_if(run_once),
        );
    }
    Ok(())
}

fn setup_campaign(app: &mut App, args: &CliArgs, config: &mut GameConfig) -> Result<(), ExitCode> {
    config.skip_mode = true;
    config.selected_gamemode = GameMode::Campaign;

    let group = &args
        .campaign_level
        .group
        .clone()
        .expect("Clap ensures that the group is specified");
    let level = &args
        .campaign_level
        .level
        .clone()
        .expect("Clap ensures that the level is specified");

    // XXX: you currently have to set both group and level to "list"
    // to be able to list all groups
    if group == "list" {
        // list all level groups
        for id in CampaignGroupWrapper::list_ids() {
            println!("{}", id);
        }
        #[cfg(target_family = "wasm")]
        quit();
        return Err(ExitCode::SUCCESS);
    } else if level == "list" {
        // attempt to list all levels in the group
        if let Some(list) = CampaignGroupWrapper::list_level_ids(group) {
            for id in list {
                println!("{}", id);
            }
            #[cfg(target_family = "wasm")]
            quit();
            return Err(ExitCode::SUCCESS);
        } else {
            eprintln!("Group '{}' does not exist", group);
            #[cfg(target_family = "wasm")]
            quit();
            return Err(ExitCode::FAILURE);
        }
    }

    // determine the wrapper level
    let wrapper = match CampaignGroupWrapper::from_ids(group, level) {
        Ok(wrapper) => wrapper,
        Err(err) => {
            // print error message and exit application
            match err {
                CampaignGroupParserError::InvalidGroup => {
                    eprintln!("Group '{}' does not exist", group)
                }
                CampaignGroupParserError::InvalidLevel => {
                    eprintln!("Level '{}' not in group '{}'", level, group)
                }
            };
            #[cfg(target_family = "wasm")]
            quit();
            return Err(ExitCode::FAILURE);
        }
    };

    let bot_support = wrapper.get_bot_type().is_some();
    app.insert_resource(CampaignModeConfig {
        level: wrapper.clone(),
    });

    if args.bot.is_some() {
        if bot_support {
            // start bot and connect it to the campaign level
            app.add_systems(
                OnEnter(AppState::AppMenu),
                (|mut commands: Commands, config: Res<CampaignModeConfig>| {
                    let level = config.level.clone();
                    commands.queue(move |w: &mut World| {
                        level.init_bot(w);
                    });
                })
                .run_if(run_once),
            );
        } else {
            // the campaign level does not support bots
            eprintln!("The selected campaign level does not support bots");
            #[cfg(target_family = "wasm")]
            quit();
            return Err(ExitCode::FAILURE);
        }
    } else {
        app.add_systems(
            OnEnter(AppState::AppMenu),
            (|mut commands: Commands| {
                commands.queue(move |w: &mut World| w.run_system_once(init_campaign_mode).unwrap());
            })
            .run_if(run_once),
        );
    }
    Ok(())
}
