use clap::{Args, Parser};
use terratactician_expandoria::game::gamemodes::GameMode;

#[derive(Args, Debug, Clone, Default)]
#[group(multiple = false)]
pub struct CliGameMode {
    /// Automatically start the game in challenge mode
    #[arg(short = 'c', long = "challenge")]
    #[arg(conflicts_with_all(["group", "level"]))]
    pub challenge: bool,
    /// Automatically start the game in creative mode
    #[arg(short = 'k', long = "creative")]
    #[arg(conflicts_with("seed"), conflicts_with_all(["group", "level"]))]
    pub creative: bool,
    /// Automatically start the game in campaign mode
    /// requires a level to be selected
    #[arg(short = 'p', long = "campaign", conflicts_with("seed"))]
    #[arg(requires("level"), requires("group"))]
    pub campaign: bool,
    /// Automatically start the game in zen mode
    #[arg(short = 'z', long = "zen")]
    #[arg(conflicts_with_all(["group", "level"]))]
    pub zen: bool,
}

impl CliGameMode {
    #[allow(dead_code)]
    pub fn get_mode(&self) -> Option<GameMode> {
        if self.challenge {
            Some(GameMode::Challenge)
        } else if self.creative {
            Some(GameMode::Creative)
        } else if self.campaign {
            Some(GameMode::Campaign)
        } else if self.zen {
            Some(GameMode::Zen)
        } else {
            None
        }
    }
}

#[derive(Args, Debug, Clone, Default)]
#[group(required = false, multiple = false)]
#[cfg(feature = "graphics")]
pub struct CliDisplayMode {
    /// Start the game in Headless mode, without window
    #[arg(short = 'd', long = "headless", conflicts_with("creative"), required = cfg!(not(feature = "graphics")))]
    #[arg(requires("bot"), requires("replay_file"))]
    pub headless: bool,
    /// Start game in Graphic mode, with a window attached
    #[arg(short = 'g', long = "graphic", default_value_t = true)]
    pub graphic: bool,
}

/// Additional configuration options for campaign mode
#[derive(Args, Debug, Clone, Default)]
pub struct CliCampaignModeConfig {
    /// the group in which the level can be found
    /// ("list" to print all available groups)
    #[arg(short = 'L', long = "group")]
    #[arg(requires("campaign"))]
    #[arg(requires("level"))]
    pub group: Option<String>,
    /// the level from the group
    /// ("list" to print all available levels in a group)
    #[arg(short = 'l', long = "level")]
    #[arg(requires("campaign"))]
    #[arg(requires("group"))]
    pub level: Option<String>,
}

#[derive(Parser, Debug, Default)]
#[command(version, about)]
pub struct CliArgs {
    /// Control how to interact with the application
    #[cfg(feature = "graphics")]
    #[command(flatten)]
    pub display_mode: CliDisplayMode,
    /// Automatically launch game in a specific game mode
    #[command(flatten)]
    pub gamemode: CliGameMode,
    /// Set the game seed.
    /// A random seed will be used if unset
    #[arg(short = 's', long = "seed")]
    #[arg(conflicts_with_all(["creative", "campaign"]))]
    pub seed: Option<String>,
    #[command(flatten)]
    pub campaign_level: CliCampaignModeConfig,
    /// Stops the game from loading and saving the config file
    #[arg(long = "no-config")]
    pub disable_config: bool,
    /// Configure a bot.
    /// When a game mode is specified the game will be started in bot mode.
    /// When --bot is used without value, the default host:port will be used: "localhost:7738".
    #[arg(
        short = 'b',
        long = "bot",
        value_name = "IP:PORT",
        default_missing_value = "localhost:7738",
        num_args = 0..=1
    )]
    pub bot: Option<String>,

    /// Sets path to recorder file and enables the recorder.
    /// If set the file will be overwritten with the record of a finished game.
    #[arg(
        short = 'r',
        long = "recorder-file",
        value_name = "path",
        conflicts_with("campaign")
    )]
    pub recorder_file: Option<String>,

    /// Nickname to save to the record file.
    #[arg(short = 'n', long = "replay-nickname", requires = "replay_file")]
    pub replay_name: Option<String>,

    /// Path to a record file. The game saved in this record will be replayed.
    #[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
    #[arg(
        short = 'y',
        long = "replay-file",
        value_name = "path",
        conflicts_with_all(["seed", "bot", "recorder_file"]), 
        required = cfg!(not(feature = "graphics"))
    )]
    pub replay_file: Option<String>,

    /// Create a report of the current game. The game results will be written to this file. When running a replay, it will be included.
    #[arg(short = 't', long = "report-file", value_name = "path")]
    pub report_file: Option<String>,
}
