pub mod coordinates;
pub mod errors;
pub mod game;
pub mod i18n;
pub mod random;
#[cfg(feature = "graphics")]
mod settings;
#[cfg(feature = "graphics")]
pub mod ui;
pub mod websocket;

#[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
use bevy::asset::AssetMetaCheck;
#[cfg(not(feature = "graphics"))]
use bevy::log::Level;
#[cfg(feature = "graphics")]
use bevy::{diagnostic::FrameTimeDiagnosticsPlugin, window::PresentMode};
use bevy::{log::LogPlugin, prelude::*, state::app::StatesPlugin};
use clap::ValueEnum;
#[cfg(feature = "graphics")]
use game::asset_loading::LACManager;
use game::{gamemodes::GameMode, GamePlugin};
#[cfg(feature = "graphics")]
use i18n::TranslationsPlugin;
#[cfg(feature = "graphics")]
use i18n::{Localization, Translate};
use serde::{Deserialize, Serialize};
#[cfg(feature = "graphics")]
use settings::ActiveSettingsBank;
#[cfg(feature = "graphics")]
use settings::{
    get_config_path, log::LogLevelWrapper, GameSettings, GameSettingsError, GameSettingsPlugin,
};
use std::hash::Hash;

#[cfg(feature = "graphics")]
use ui::UiPlugin;

// import wasm_bindgen to define extern wasm function.
#[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
use wasm_bindgen::prelude::*;

#[derive(Debug, Default, Clone, Copy, PartialEq, Eq, Hash, States)]
pub enum AppState {
    /// Asset loading screen
    /// starts loading assets
    /// exists when all required assets were loaded
    #[default]
    LoadApp,
    /// the user is in the app menu
    /// but not configuring the game
    AppMenu,
    /// Game configuration screen
    /// allows changing the seed, difficulty,...
    SetupMenu,
    /// The player is currently playing the game
    /// See GameState for more specific states
    InGame,
    /// Error screen
    /// see ErrorDisplay for more
    Error,
}

#[derive(SubStates, Default, Debug, Hash, PartialEq, Eq, Clone)]
#[source(AppState = AppState::AppMenu)]
pub enum AppMenuState {
    /// The Welcome screen is shown
    #[default]
    MainMenu,
    /// Settings Menu is opened
    /// see SettingsTab for more
    SettingsMenu,
    /// About Screen
    /// shows app & license info
    About,
}

#[derive(States, Debug, Clone, Copy, Hash, Eq, PartialEq, Default, ValueEnum)]
pub enum DisplayMode {
    /// Headless session
    /// doesn't spawn a window
    Headless,
    /// Graphical session
    /// a window is shown and the Game is rendered in 3D
    /// requires assets to be loaded
    #[default]
    Graphic,
}

// If APP_ID gets changed, the package.metadata.deb section in Cargo.toml must be rewritten.
#[cfg(feature = "graphics")]
const APP_ID: &str = "terratactician-expandoria";
#[cfg(feature = "graphics")]
const VERSION: &str = env!("CARGO_PKG_VERSION");

pub struct AppBuilder {
    /// Specifies how the user interacts with the game
    pub display_mode: DisplayMode,
    /// Enables loading and writing of config file from disk
    #[cfg(feature = "graphics")]
    pub enable_config_io: bool,
    /// Enable registration of the LogPlugin
    /// Note: This may only be done once per session
    pub register_logger: bool,
}
impl Default for AppBuilder {
    fn default() -> Self {
        Self {
            display_mode: DisplayMode::Graphic,
            #[cfg(feature = "graphics")]
            enable_config_io: true,
            register_logger: true,
        }
    }
}

impl AppBuilder {
    pub fn build(&self) -> App {
        let mut app = App::new();

        // load settings
        // required by logger
        #[cfg(feature = "graphics")]
        let config = if !self.enable_config_io {
            info!("Loading config file was disabled by user");
            GameSettings::default()
        } else if let Some(config_file) = get_config_path() {
            info!("Checking for config at: {}", config_file);
            let config = GameSettings::from_file(&config_file);
            match config {
                Ok(conf) => {
                    info!("Loading config: {:?}", conf);
                    conf
                }
                Err(msg) => {
                    let settings = GameSettings::default();
                    match msg {
                        GameSettingsError::ConfigDoesNotExist => {
                            // the user doesn't have a config file
                            // generate one now, so they don't have to start from scratch
                            match settings.to_file(&config_file) {
                                Ok(()) => info!("Bootstrapped config file"),
                                Err(msg) => error!("Couldn't generate config file: {:?}", msg),
                            }
                        }
                        _ => error!("Couldn't load config file: {:?}", msg),
                    }
                    settings
                }
            }
        } else {
            error!("No config file path available. Settings won't be saved");
            GameSettings::default()
        };

        #[cfg(feature = "graphics")]
        let window_conf = WindowPlugin {
            primary_window: Some(Window {
                title: "TerraTactician - Expandoria".to_string(),
                // todo change Window-Icon, maybe like this:
                // https://bevy-cheatbook.github.io/window/icon.html

                // todo change default values
                resolution: (800., 600.).into(),
                present_mode: if config.graphics.vsync {
                    PresentMode::AutoVsync
                } else {
                    PresentMode::AutoNoVsync
                },
                resize_constraints: WindowResizeConstraints {
                    min_height: MIN_WIN_SIZE_HEIGHT,
                    min_width: MIN_WIN_SIZE_WIDTH,
                    max_height: f32::INFINITY,
                    max_width: f32::INFINITY,
                },
                canvas: Some("#bevy".to_owned()),
                prevent_default_event_handling: true, // needed for touch
                ..default()
            }),
            ..default()
        };

        #[cfg(feature = "graphics")]
        let mut default_plugin = match self.display_mode {
            DisplayMode::Graphic => DefaultPlugins
                .build()
                .set(window_conf)
                .disable::<LogPlugin>(),
            DisplayMode::Headless => MinimalPlugins
                .build()
                .add(StatesPlugin)
                .add(AssetPlugin::default()),
        };
        #[cfg(not(feature = "graphics"))]
        let default_plugin = MinimalPlugins
            .build()
            .add(StatesPlugin)
            .add(AssetPlugin::default());

        if self.register_logger {
            #[cfg(feature = "graphics")]
            match config.log.level {
                LogLevelWrapper::Level(level) => {
                    // only use the file logger when the file-logger feature is enabled
                    // this disables the stdout logger
                    // NOTE: cannot use the cfg! macro here as this section has optional dependencies
                    #[cfg(all(
                        feature = "file-logger",
                        not(any(target_arch = "wasm32", target_arch = "wasm64"))
                    ))]
                    {
                        use bevy::log::tracing_subscriber;
                        use settings::get_data_dir;
                        app.add_plugins(LogPlugin {
                            // apply log level
                            level,
                            custom_layer: |_| {
                                Some({
                                    // read log output directory from TTE_LOG_DIR environment variable
                                    // if unset use the application data directory
                                    // in case this is unavailable use the current directory
                                    let data_dir = std::env::var("TTE_LOG_DIR")
                                        .unwrap_or(get_data_dir().unwrap_or(String::from("./")));
                                    println!("Using {} as log directory", data_dir);
                                    // create a file appender
                                    // this can be used for write operations and supports log rotataion
                                    let appender = tracing_appender::rolling::hourly(
                                        data_dir,
                                        &format!("{}.log", APP_ID),
                                    );
                                    // to be able make use of the appender, we have to create a new tracing subscriber
                                    use bevy::log::tracing_subscriber::Layer;
                                    tracing_subscriber::fmt::layer()
                                        .with_writer(appender)
                                        .boxed()
                                })
                            },
                            ..Default::default()
                        });
                    }
                    #[cfg(any(
                        not(feature = "file-logger"),
                        target_arch = "wasm32",
                        target_arch = "wasm64"
                    ))]
                    {
                        app.add_plugins(LogPlugin {
                            // apply log level
                            level,
                            ..Default::default()
                        });
                    }
                }
                // nothing to do as the LogPlugin is not registered
                LogLevelWrapper::Disabled => (),
            }
            #[cfg(not(feature = "graphics"))]
            {
                app.add_plugins(LogPlugin {
                    level: Level::INFO,
                    ..Default::default()
                });
            }
        }

        // setting assets path. If either APP_ID or the path gets changed, the package.metadata.deb section in Cargo.toml must be rewritten.
        #[cfg(feature = "custom-assets-path")]
        let build_assets_path: String = option_env!("TTE_BUILD_ASSETS_DIR")
            .map(|e| e.to_string())
            .unwrap_or(format!("/usr/share/{}/assets", APP_ID));

        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display_mode {
            default_plugin = default_plugin.set(AssetPlugin {
                #[cfg(feature = "custom-assets-path")]
                file_path: build_assets_path,
                // fixes asset loading when using trunk serve
                // because it doesn't send 404 when a file isn't found
                // causing bevy to try and load the asset meta files
                #[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
                meta_check: AssetMetaCheck::Never,
                ..default()
            });
        }

        app.add_plugins(default_plugin);

        // the AppState has the added to the app before connecting scoped entities to it
        app.init_state::<AppState>();

        // create an additional display mode state
        // to toggle specific systems on and off
        // without having to pass display around
        app.insert_state(self.display_mode);

        // Plugin loads assets and waits for them.
        app.add_plugins(game::asset_loading::AppLoadingPlugin);

        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display_mode {
            app.add_plugins(FrameTimeDiagnosticsPlugin)
                .add_plugins(UiPlugin)
                .add_plugins(TranslationsPlugin);

            app.register_asset_collection::<AppIcon>();

            app.add_systems(
                Update,
                update_vsync.run_if(resource_changed::<ActiveSettingsBank>),
            );
            app.add_sub_state::<AppMenuState>();
        }

        app.add_plugins(GamePlugin::new(self.display_mode));
        #[cfg(feature = "graphics")]
        app.add_plugins(GameSettingsPlugin {
            disk_config: config,
            write_to_disk: self.enable_config_io,
        });

        app.enable_state_scoped_entities::<AppState>();

        app
    }
}

#[derive(Debug, Resource, Default)]
pub struct GameConfig {
    pub selected_gamemode: GameMode,
    pub bot_url: String,
    pub use_bot: bool,
    pub record_game: Option<String>,
    pub replay_name: String,
    pub activate_recorder: bool,

    /// is some a report will be saved on the end of a game
    pub report_file: Option<String>,

    /// skips next load from settings
    pub skip_seed: bool,
    /// skips next load from settings
    pub skip_mode: bool,
    /// skips next load from settings
    pub skip_bot: bool,
    /// skips next load from settings
    pub skip_recorder: bool,
}

#[derive(Default, Debug, Clone, Serialize, Deserialize, PartialEq)]
pub enum Difficulty {
    Easy,
    #[default]
    Medium,
    Hard,
}

#[cfg(feature = "graphics")]
impl Difficulty {
    pub fn get_name(&self, localization: &Localization) -> String {
        match self {
            Difficulty::Easy => localization
                .translate("setup_menu-difficulty-easy")
                .to_string(),
            Difficulty::Medium => localization
                .translate("setup_menu-difficulty-medium")
                .to_string(),
            Difficulty::Hard => localization
                .translate("setup_menu-difficulty-hard")
                .to_string(),
        }
    }
    pub fn list() -> Vec<Self> {
        vec![Self::Easy, Self::Medium, Self::Hard]
    }
}

// todo: in one
#[cfg(feature = "graphics")]
const MIN_WIN_SIZE_WIDTH: f32 = if cfg!(target_arch = "wasm32") {
    150.0
} else {
    600.0
};
#[cfg(feature = "graphics")]
const MIN_WIN_SIZE_HEIGHT: f32 = if cfg!(target_arch = "wasm32") {
    100.0
} else {
    400.0
};

#[cfg(feature = "graphics")]
fn update_vsync(mut windows: Query<&mut Window>, settings: Res<ActiveSettingsBank>) {
    for mut window in windows.iter_mut() {
        window.present_mode = if settings.graphics.vsync {
            PresentMode::AutoVsync
        } else {
            PresentMode::AutoNoVsync
        };
    }
}

#[cfg(feature = "graphics")]
define_asset_collection!(
    AppIcon,
    ?icon : Image = "icons/app-icon.png",
);

#[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
#[wasm_bindgen(module = "/src/tools.js")]
extern "C" {
    /// Use history.back() with fallback hack in js.
    pub fn quit();
    pub fn download(data: &str, file: &str, mime_type: &str);
}

pub mod prelude {
    #[cfg(feature = "graphics")]
    use bevy::math::Vec2;
    use bevy::prelude::{NextState, Res};
    use bevy::state::state::{FreelyMutableState, State, States};
    use bevy::time::{Real, Time, Timer, TimerMode};
    #[cfg(feature = "graphics")]
    use bevy_egui::egui;
    use std::time::Duration;

    pub use crate::game::asset_loading::LACManager;
    pub use crate::game::resources::HumanReadableNumber;
    #[cfg(feature = "graphics")]
    pub use crate::i18n::Translate;
    #[cfg(feature = "graphics")]
    pub use crate::settings::keybindings::WithTranslationID;

    pub trait Cast<T> {
        fn cast(self) -> T;
    }

    #[cfg(feature = "graphics")]
    impl Cast<egui::Vec2> for Vec2 {
        fn cast(self) -> egui::Vec2 {
            egui::Vec2::new(self.x, self.y)
        }
    }
    #[cfg(feature = "graphics")]
    impl Cast<bevy::color::Color> for egui::Color32 {
        fn cast(self) -> bevy::color::Color {
            bevy::color::Color::srgba_u8(self.r(), self.g(), self.b(), self.a())
        }
    }

    /// Generates a [`Condition`](bevy_ecs::prelude::Condition)-satisfying closure that returns `true`
    /// if the state machine is currently in one of the `states`.
    ///
    /// Will return `false` if the state does not exist or if not in `state`.
    pub fn in_state_any<S: States>(
        states: Vec<S>,
    ) -> impl FnMut(Option<Res<State<S>>>) -> bool + Clone {
        move |current_state: Option<Res<State<S>>>| match current_state {
            Some(current_state) => states.contains(&current_state),
            None => false,
        }
    }

    /// Run condition that is active on a regular time interval, using `Time<Real>` to advance
    /// the timer.
    ///
    /// Note that this does **not** guarantee that systems will run at exactly the
    /// specified interval. If delta time is larger than the specified `duration` then
    /// the system will only run once even though the timer may have completed multiple
    /// times. This condition should only be used with large time durations (relative to
    /// delta time).
    ///
    /// derived from bevy's on_timer
    pub fn on_timer_real(duration: Duration) -> impl FnMut(Res<Time<Real>>) -> bool + Clone {
        let mut timer = Timer::new(duration, TimerMode::Repeating);
        move |time: Res<Time<Real>>| {
            timer.tick(time.delta());
            timer.just_finished()
        }
    }

    /// Generates a [`Condition`](bevy_ecs::prelude::Condition)-satisfying closure that returns `true`
    /// if the state machine is currently in `state` and no other next state was requested.
    ///
    /// Will return `false` if the state does not exists, the next state does not exists, or if not in `state` or another state was requested before.
    #[allow(clippy::type_complexity)]
    pub fn still_in_state<S: States + FreelyMutableState>(
        state: S,
    ) -> impl FnMut(Option<Res<State<S>>>, Option<Res<NextState<S>>>) -> bool + Clone {
        move |current_state: Option<Res<State<S>>>, next_state: Option<Res<NextState<S>>>| match (
            current_state,
            next_state.as_ref().map(|i| i.as_ref()),
        ) {
            (Some(current_state), Some(NextState::Unchanged)) => state == **current_state,
            (Some(current_state), Some(NextState::Pending(next_state))) => {
                state == **current_state && state == *next_state
            }
            _ => false,
        }
    }

    /// Generates a [`Condition`](bevy_ecs::prelude::Condition)-satisfying closure that always returns `false`
    pub fn never() -> impl FnMut() -> bool + Clone {
        move || false
    }
}
