use bevy::{
    asset::{Asset, AssetLoader},
    reflect::TypePath,
};
use fluent::FluentResource;
use thiserror::{self, Error};

#[derive(Asset, TypePath)]
pub struct FluentResourceAsset(pub FluentResource);

#[derive(Debug, Error)]
#[non_exhaustive]
pub enum FluentResourceLoadingError {
    #[error("Could not load asset: {0}")]
    Io(#[from] std::io::Error),
    #[error("Could not parse utf8 file: {0}")]
    Utf8(#[from] std::string::FromUtf8Error),
    #[error("Could not parse fluent translation file")]
    Fluent(),
}

#[derive(Default)]
pub struct FluentResourceLoader;
impl AssetLoader for FluentResourceLoader {
    type Asset = FluentResourceAsset;
    type Settings = ();
    type Error = FluentResourceLoadingError;

    async fn load(
        &self,
        reader: &mut dyn bevy::asset::io::Reader,
        _settings: &(),
        _load_context: &mut bevy::asset::LoadContext<'_>,
    ) -> Result<Self::Asset, Self::Error> {
        let mut bytes = Vec::new();
        reader.read_to_end(&mut bytes).await?;
        let text = String::from_utf8(bytes)?;
        match FluentResource::try_new(text) {
            Ok(res) => Ok(FluentResourceAsset(res)),
            Err((_, _errors)) => Err(FluentResourceLoadingError::Fluent()),
        }
    }

    fn extensions(&self) -> &[&str] {
        &[]
    }
}
