use crate::{
    game::{
        tiles::rgb_tile::{update_tile_data, RgbTileData},
        visuals::debug_text::TextEntityData,
    },
    AppState,
};
use bevy::prelude::*;

pub struct RgbTilePlugin;
impl Plugin for RgbTilePlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            Update,
            (material_add, material_update.after(update_tile_data))
                .chain()
                .run_if(in_state(AppState::InGame)),
        );
    }
}

/// adjusts the tile color to match the tile color data
fn material_update(
    tiles: Query<(Entity, &RgbTileData), Changed<RgbTileData>>,
    children: Query<&Children>,
    objects: Query<&MeshMaterial3d<StandardMaterial>, Without<TextEntityData>>,
    mut standard_materials: ResMut<Assets<StandardMaterial>>,
) {
    for (entity, color) in &tiles {
        for entity in children.iter_descendants(entity) {
            let Ok(handle) = objects.get(entity) else {
                continue;
            };
            let Some(std_mat) = standard_materials.get_mut(handle) else {
                continue;
            };

            std_mat.base_color = color.into();
        }
    }
}
/// replaces the tiles StandardMaterial, so that they can be modified individually
fn material_add(
    tiles: Query<(Entity, &RgbTileData), Added<RgbTileData>>,
    children: Query<&Children>,
    mut standard_materials: ResMut<Assets<StandardMaterial>>,
    mut commands: Commands,
) {
    for (entity, color) in &tiles {
        for entity in children.iter_descendants(entity) {
            let color: Color = color.into();

            commands
                .entity(entity)
                .insert(Visibility::Visible)
                .insert(MeshMaterial3d(standard_materials.add(color)));
        }
    }
}
