use bevy::{
    pbr::{ExtendedMaterial, MaterialExtension},
    prelude::*,
    render::render_resource::{AsBindGroup, ShaderRef},
};

use crate::{
    define_asset_collection,
    game::{
        placement_reclaim::{PlacementTimer, TileLoaded},
        tiles::TileType,
        time::{Game, GameTime},
    },
    prelude::LACManager,
    AppState,
};

use super::debug_text::TextEntityData;

pub struct PlacementAnimationPlugin;

impl Plugin for PlacementAnimationPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<PlacmenAnimationShader>();
        app.add_plugins((MaterialPlugin::<
            ExtendedMaterial<StandardMaterial, LoadingMaterial>,
        >::default(),));
        app.add_systems(
            Update,
            (
                set_animation_material,
                // moved to apply_placement_timer.before(set_animation_material)
                // .after(apply_placement_timer),
                reset_material,
            )
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(
            PostUpdate,
            update_material.run_if(in_state(AppState::InGame)),
        );
    }
}

type AnimationMaterial = ExtendedMaterial<StandardMaterial, LoadingMaterial>;

#[allow(clippy::too_many_arguments)]
pub fn set_animation_material(
    mut commands: Commands,
    tiles: Query<(Entity, &TileType, &PlacementTimer, &Transform), Added<PlacementTimer>>,
    children: Query<&Children>,
    objects: Query<&MeshMaterial3d<StandardMaterial>, Without<TextEntityData>>,
    standard_materials: Res<Assets<StandardMaterial>>,
    time: Res<Time<Virtual>>,
    gtime: Res<Time<Game>>,
    mut extended_materials: ResMut<Assets<AnimationMaterial>>,
) {
    for (entity, tile_type, timer, transform) in &tiles {
        for entity in children.iter_descendants(entity) {
            let Ok(handle) = objects.get(entity) else {
                continue;
            };

            let Some(std_mat) = standard_materials.get(handle) else {
                continue;
            };

            let material = extended_materials.add(ExtendedMaterial {
                base: std_mat.clone(),
                extension: LoadingMaterial::new(
                    time.elapsed_secs_wrapped()
                        - timer.timer.elapsed_secs()
                        // adjust for game time (reclaim starts on begin of resource tick)
                        - gtime.sub_secs(),
                    timer.timer.duration().as_secs_f32(),
                    tile_type.get_tile_height(),
                    transform.translation.y,
                ),
            });

            commands
                .entity(entity)
                .remove::<MeshMaterial3d<StandardMaterial>>()
                .insert(MeshMaterial3d(material.clone()));
        }
    }
}

fn reset_material(
    mut commands: Commands,
    tiles: Query<Entity, Added<TileLoaded>>,
    children: Query<&Children>,
    objects: Query<&MeshMaterial3d<AnimationMaterial>>,
    mut standard_materials: ResMut<Assets<StandardMaterial>>,
    extended_materials: Res<Assets<AnimationMaterial>>,
) {
    for entity in &tiles {
        for entity in children.iter_descendants(entity) {
            let Ok(handle) = objects.get(entity) else {
                continue;
            };

            let ext_mat = extended_materials
                .get(handle)
                .expect("The Material has to be registered.");

            let std_mat_handle = standard_materials.add(ext_mat.base.clone());

            commands
                .entity(entity)
                .remove::<MeshMaterial3d<AnimationMaterial>>()
                .insert(MeshMaterial3d(std_mat_handle));
        }
    }
}

#[allow(clippy::type_complexity)]
fn update_material(
    mut extended_materials: ResMut<Assets<AnimationMaterial>>,
    tiles: Query<(Entity, &Transform), (Changed<Transform>, With<TileType>)>,
    children: Query<&Children>,
    objects: Query<&MeshMaterial3d<AnimationMaterial>>,
) {
    for (entity, transform) in &tiles {
        for entity in children.iter_descendants(entity) {
            let Ok(handle) = objects.get(entity) else {
                continue;
            };
            let ext_mat = extended_materials
                .get_mut(handle)
                .expect("The Material has to be registered.");
            ext_mat.extension.set_base_height(transform.translation.y);
        }
    }
}

#[derive(Asset, AsBindGroup, Reflect, Debug, Clone)]
pub struct LoadingMaterial {
    /// hack to get var 4 Byte alligned: start, duration, height
    #[uniform(100)]
    data: Vec4,
}

impl LoadingMaterial {
    fn new(start: f32, duration: f32, height: f32, base_height: f32) -> Self {
        Self {
            data: Vec4::new(start, duration, height, base_height),
        }
    }
    fn set_base_height(&mut self, height: f32) {
        self.data.w = height;
    }
}

define_asset_collection!(
    PlacmenAnimationShader,
    !shader : Shader = "shaders/placement_animation.wgsl",
);

impl MaterialExtension for LoadingMaterial {
    fn fragment_shader() -> ShaderRef {
        "shaders/placement_animation.wgsl".into()
    }
    fn deferred_fragment_shader() -> ShaderRef {
        "shaders/placement_animation.wgsl".into()
    }
}
