use crate::{prelude::Cast, settings::ActiveSettingsBank, ui::theme::BACKGROUND_COLOR, AppState};
use bevy::{
    pbr::{CascadeShadowConfig, CascadeShadowConfigBuilder},
    prelude::*,
};
use std::f32::consts::PI;

pub struct LightPlugin;
impl Plugin for LightPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_systems(OnEnter(AppState::InGame), setup_light);
        app.add_systems(OnExit(AppState::InGame), despawn_light);
        app.add_systems(
            Update,
            update_light_settings
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::InGame)),
        );
    }
}

#[derive(Component)]
pub struct Sun;

/// Static Sun is a dimm static directional light, used to preserve contrast.
#[derive(Component)]
pub struct StaticSun;

pub fn setup_light(mut commands: Commands, settings: Res<ActiveSettingsBank>) {
    //#f5e0dc
    commands.insert_resource(ClearColor(BACKGROUND_COLOR.cast()));

    commands.insert_resource(AmbientLight {
        color: Color::WHITE,
        brightness: 300.0,
    });

    commands.spawn((
        Transform::from_rotation(Quat::from_euler(
            EulerRot::ZYX,
            15. / 180. * PI,
            0.0,
            -PI / 3.,
        )),
        DirectionalLight {
            color: Color::srgb(1.0, 1.0, 0.8),
            shadows_enabled: settings.graphics.shadows.enabled(),
            illuminance: 2000.0,
            ..default()
        },
        CascadeShadowConfigBuilder {
            num_cascades: settings.graphics.shadows.get_cascades(),
            maximum_distance: settings.graphics.shadows.get_max_distance(),
            ..default()
        }
        .build(),
        Sun,
    ));

    // Web only supports one directional light.
    #[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
    commands.spawn((
        Transform::from_rotation(Quat::from_euler(EulerRot::ZYX, 0.0, 0.0, -PI / 2.)),
        DirectionalLight {
            color: Color::WHITE,
            shadows_enabled: false,
            illuminance: 200.0,
            ..default()
        },
        StaticSun,
    ));
}

#[allow(clippy::type_complexity)]
pub fn despawn_light(
    mut commands: Commands,
    suns: Query<Entity, Or<(With<Sun>, With<StaticSun>)>>,
) {
    commands.remove_resource::<ClearColor>();
    commands.remove_resource::<AmbientLight>();
    for s in suns.iter() {
        commands.entity(s).despawn_recursive();
    }
}

pub fn update_light_settings(
    settings: Res<ActiveSettingsBank>,
    mut sun: Query<(&mut DirectionalLight, &mut CascadeShadowConfig), With<Sun>>,
) {
    for (mut light, mut shadow) in sun.iter_mut() {
        light.shadows_enabled = settings.graphics.shadows.enabled();
        *shadow = CascadeShadowConfigBuilder {
            num_cascades: settings.graphics.shadows.get_cascades(),
            maximum_distance: settings.graphics.shadows.get_max_distance(),
            ..default()
        }
        .into();
    }
}
