use bevy::prelude::*;
use bevy_enum_filter::Enum;

use super::debug_text::{spawn_debug_text, DebugText};
use crate::{
    game::tiles::{
        assets::TileModels,
        chameleon_tile::{update_tile_data, ChameleonTileData, ChameleonTileUpdate},
        tile_type_filters, TileScene,
    },
    AppState,
};

pub struct ChameleonTilePlugin;
impl Plugin for ChameleonTilePlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            Update,
            (
                material_add,
                material_update
                    .after(update_tile_data)
                    .before(spawn_debug_text),
            )
                .chain()
                .run_if(in_state(AppState::InGame)),
        );
    }
}

/// adjusts tile model and updates debug text to match the tiles data
#[allow(clippy::type_complexity)]
fn material_update(
    mut commands: Commands,
    mut tiles: Query<
        (
            Entity,
            &ChameleonTileData,
            &mut ChameleonTileUpdate,
            &mut DebugText,
            &Children,
        ),
        Changed<ChameleonTileData>,
    >,
    models: Res<TileModels>,
    tile_assets: Res<Assets<Gltf>>,
    scenes: Query<Entity, With<TileScene>>,
) {
    for (entity, data, mut update, mut text, childs) in tiles.iter_mut() {
        if update.text {
            text.set_text(data.text.clone());
            update.text = false;
        }
        if update.model {
            // remove old scene
            for child in childs {
                if let Ok(id) = scenes.get(*child) {
                    commands.entity(id).despawn_recursive();
                }
            }

            // load new scene
            let scene = {
                let model_handle = models.0.get(&data.model).expect(
                    "The Loading Screen should have taken care of loading the model at this point.",
                );

                let gltf = tile_assets
                    .get(model_handle)
                    .expect("The LoadingScreen should have ensured that the asset was loaded.");

                gltf.scenes
                    .get(data.model.get_asset_scene())
                    .expect("The Scene should be available for the model")
                    .clone_weak()
            };

            commands
                .entity(entity)
                .with_child((TileScene, SceneRoot(scene), Transform::default()));
            text.height = data.model.get_tile_height();
            update.model = false;
        }
    }
}

/// replaces the scene with the proper model
#[allow(clippy::type_complexity)]
fn material_add(
    mut commands: Commands,
    scenes: Query<Entity, With<TileScene>>,
    models: Res<TileModels>,
    tile_assets: Res<Assets<Gltf>>,
    mut tiles: Query<
        (Entity, &ChameleonTileData, &mut DebugText, &Children),
        (With<Enum!(TileType::Chameleon)>, Added<ChameleonTileData>),
    >,
) {
    for (entity, data, mut text, childs) in tiles.iter_mut() {
        text.set_text(data.text.clone());
        // remove old scene
        for child in childs {
            if let Ok(id) = scenes.get(*child) {
                commands.entity(id).despawn_recursive();
            }
        }

        // load new scene
        let scene = {
            let model_handle = models.0.get(&data.model).expect(
                "The Loading Screen should have taken care of loading the model at this point.",
            );

            let gltf = tile_assets
                .get(model_handle)
                .expect("The LoadingScreen should have ensured that the asset was loaded.");

            gltf.scenes
                .get(data.model.get_asset_scene())
                .expect("The Scene should be available for the model")
                .clone_weak()
        };

        commands
            .entity(entity)
            .with_child((TileScene, SceneRoot(scene), Transform::default()))
            .insert(Visibility::Visible);
        text.height = data.model.get_tile_height();
    }
}
