use crate::{
    coordinates::{CubeCoordinate, EDGE_LENGTH, TILE_SIZE},
    game::build_area::{BuildArea, BUILD_PLATE_HEIGHT},
    settings::ActiveSettingsBank,
    ui::DebugMaterial,
    AppState,
};
use bevy::{
    prelude::*,
    render::{
        mesh::{Indices, PrimitiveTopology},
        render_asset::RenderAssetUsages,
    },
};

const BUILD_GRID_COLOR_BARE: Color = Color::srgb(1., 1., 1.);
const BUILD_GRID_COLOR_SUBMERGED: Color = Color::srgb(0., 0., 0.5);

pub struct BuildGridPlugin;

impl Plugin for BuildGridPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_systems(Update, setup_build_grid.run_if(in_state(AppState::InGame)));
        app.add_systems(
            Update,
            update_build_grid
                .after(setup_build_grid)
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(
            Update,
            update_build_grid_settings.run_if(resource_changed::<ActiveSettingsBank>),
        );
    }
}

fn setup_build_grid(
    mut commands: Commands,
    mut materials: ResMut<Assets<StandardMaterial>>,
    mut meshes: ResMut<Assets<Mesh>>,
    build_area: Query<(Entity, &BuildArea), Added<BuildArea>>,
    settings: Res<ActiveSettingsBank>,
) {
    let material = materials.add(StandardMaterial {
        base_color: if settings.graphics.water {
            BUILD_GRID_COLOR_SUBMERGED
        } else {
            BUILD_GRID_COLOR_BARE
        },
        reflectance: 0.0,
        metallic: 0.0,
        perceptual_roughness: 1.0,
        specular_transmission: 0.0,
        diffuse_transmission: 0.0,
        ..default()
    });

    for (build_area_entity, build_area) in build_area.iter() {
        let positions: Vec<[f32; 3]> = vec![];

        let mesh = Mesh::new(
            PrimitiveTopology::TriangleList,
            RenderAssetUsages::default(),
        )
        .with_inserted_attribute(Mesh::ATTRIBUTE_POSITION, positions);
        let mesh_h = meshes.add(mesh);

        commands.entity(build_area_entity).insert((
            Mesh3d(mesh_h.clone()),
            MeshMaterial3d(material.clone()),
            Transform::from_translation(build_area.center.into()),
            DebugMaterial("Build Grid"),
        ));
    }
}

fn update_build_grid_settings(
    material: Query<&MeshMaterial3d<StandardMaterial>, With<BuildArea>>,
    settings: Res<ActiveSettingsBank>,
    mut materials: ResMut<Assets<StandardMaterial>>,
) {
    for material in material.iter() {
        if let Some(material) = materials.get_mut(material) {
            material.base_color = if settings.graphics.water {
                BUILD_GRID_COLOR_SUBMERGED
            } else {
                BUILD_GRID_COLOR_BARE
            };
        }
    }
}

fn update_build_grid(
    mut build_area: Query<(&BuildArea, &mut Transform, &Mesh3d), Changed<BuildArea>>,
    mut meshes: ResMut<Assets<Mesh>>,
) {
    const WIDTH: f32 = 0.02 / 2.0;

    let Ok((build_area, mut transform, mesh)) = build_area.get_single_mut() else {
        return;
    };

    // move mesh to center
    transform.translation = build_area.center.into();
    transform.translation.y += BUILD_PLATE_HEIGHT;

    let Some(mesh) = meshes.get_mut(mesh) else {
        return;
    };

    // preallocate vectors with correct size
    let fields = 3 * build_area.radius * (build_area.radius + 1) + 1;
    let mut vertices_positions: Vec<[f32; 3]> = Vec::with_capacity(fields * 4 * 6);
    let mut indices: Vec<u32> = Vec::with_capacity(fields * 6 * 6);

    let mut id = 0;

    fn rotate_vec_by_60_degrees(v: &mut Vec3) {
        const RADIANS_60_COS: f32 = 0.5; //(60.0 * PI / 180.0).sin();
        const RADIANS_60_SIN: f32 = 0.866_025_4_f32; //(60.0 * PI / 180.0).cos();
        let x = v.x;
        v.x = v.x * RADIANS_60_COS + v.z * RADIANS_60_SIN;
        v.z = -x * RADIANS_60_SIN + v.z * RADIANS_60_COS;
    }

    // coords of the lower inner boarder of a tile in 0, 0, 0
    let mut v1 = Vec3::new(TILE_SIZE as f32 * 0.5, 0., EDGE_LENGTH as f32 * 0.5);
    let mut v2 = Vec3::new(TILE_SIZE as f32 * 0.5, 0., -EDGE_LENGTH as f32 * 0.5);
    let mut v3 = Vec3::new(
        TILE_SIZE as f32 * 0.5 - WIDTH,
        0.,
        -EDGE_LENGTH as f32 * 0.5,
    );
    let mut v4 = Vec3::new(TILE_SIZE as f32 * 0.5 - WIDTH, 0., EDGE_LENGTH as f32 * 0.5);

    // iterate all 6 sides, to add the boarder
    for _ in 0..6 {
        // iterate over all fields in area.
        // does not uses the .get_area iterator to get center dot.
        for ring in 0..build_area.radius {
            for coord in CubeCoordinate::new(0, 0, 0).get_ring(ring) {
                // adds the inner boarder to the current side.
                let pos: Vec3 = coord.into();
                // add verticies, shifted to the current field.
                vertices_positions.push([pos.x + v1.x, pos.y, pos.z + v1.z]);
                vertices_positions.push([pos.x + v2.x, pos.y, pos.z + v2.z]);
                vertices_positions.push([pos.x + v3.x, pos.y, pos.z + v3.z]);
                vertices_positions.push([pos.x + v4.x, pos.y, pos.z + v4.z]);
                // added vertex indices, to triangulate rectangle.
                let triangulation = [id, id + 1, id + 2, id, id + 2, id + 3];
                indices.extend_from_slice(&triangulation);
                id += 4;
            }
        }
        // rotate boarder to next side
        rotate_vec_by_60_degrees(&mut v1);
        rotate_vec_by_60_degrees(&mut v2);
        rotate_vec_by_60_degrees(&mut v3);
        rotate_vec_by_60_degrees(&mut v4);
    }

    // generate normals: all up
    let normals: Vec<[f32; 3]> = vec![[0., 1., 0.]; vertices_positions.len()];

    // update mesh
    mesh.insert_attribute(Mesh::ATTRIBUTE_POSITION, vertices_positions);
    mesh.insert_attribute(Mesh::ATTRIBUTE_NORMAL, normals);
    mesh.insert_indices(Indices::U32(indices))
}
