use std::{f32::consts::PI, time::Duration};

use bevy::{app::Plugin, gltf::Gltf, prelude::*, time::common_conditions::on_timer};
use bevy_enum_filter::Enum;
use fastrand::Rng;

use crate::{
    coordinates::CubeCoordinate,
    define_asset_collection,
    game::{
        asset_loading::LACManager,
        build_area::BuildArea,
        tiles::{tile_type_filters, TileType},
    },
    random::TTERand,
    settings::ActiveSettingsBank,
    AppState,
};

use super::animations::AnimationEntityLink;

const BEE_SPAWN_TEST_INTERVAL: f32 = 11.0;
const MAX_BEES_PER_TILE: f32 = 0.05;
const MAX_BEE_AGE: f32 = 30.0;
const BEE_RANDOM_MOVE_FACTOR: f32 = 5.0;
const BEE_MOVE_DAMPING: f32 = 0.5;

define_asset_collection!(
    BeeAsset,
    !bee : Gltf = "models/bee.glb",
);

pub struct BeePlugin;

impl Plugin for BeePlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.register_asset_collection::<BeeAsset>();
        app.add_systems(
            Update,
            spawn_despawn_bee
                .run_if(in_state(AppState::InGame))
                .run_if(on_timer(Duration::from_secs_f32(BEE_SPAWN_TEST_INTERVAL))),
        );
        app.add_systems(OnExit(AppState::InGame), despawn_bee);
        app.add_systems(Update, animate_bee.run_if(in_state(AppState::InGame)));
        app.add_systems(Update, fly_bee.run_if(in_state(AppState::InGame)));
        app.add_systems(
            Update,
            update_bee_settings
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::InGame)),
        );
    }
}
impl BeePlugin {
    /// returns the scene index
    fn get_model() -> usize {
        0
    }
    /// returns the animation index
    fn get_animation() -> usize {
        0
    }
}

#[derive(Component)]
struct Bee {
    pub origin: CubeCoordinate,
    pub vel: Vec3,
    pub age: f32,
    pub rot: f32,
}

impl Bee {
    fn new(origin: CubeCoordinate) -> Self {
        Self {
            age: 0.0,
            origin,
            vel: Vec3::new(
                fastrand::f32() * 0.5 - 0.25,
                0.0,
                fastrand::f32() * 0.5 - 0.25,
            )
            .normalize_or_zero(),
            rot: 1.0,
        }
    }
}

fn update_bee_settings(
    mut commands: Commands,
    settings: Res<ActiveSettingsBank>,
    bees: Query<Entity, With<Bee>>,
) {
    if !settings.graphics.animations {
        for bee in bees.iter() {
            commands.entity(bee).despawn_recursive();
        }
    }
}

/// spawn and despawn bees
#[allow(clippy::type_complexity)]
fn spawn_despawn_bee(
    mut commands: Commands,
    bees: Query<(Entity, &Bee)>,
    assets: Res<Assets<Gltf>>,
    collection: Res<BeeAsset>,
    bee_spawning_tiles: Query<
        (&CubeCoordinate, &TileType),
        Or<(
            With<Enum!(TileType::Forest)>,
            With<Enum!(TileType::Beehive)>,
        )>,
    >,
    build_area: Query<&BuildArea>,
    settings: Res<ActiveSettingsBank>,
) {
    if !settings.graphics.animations {
        return;
    }
    // despawn old bees
    for (entity, bee) in bees.iter() {
        if bee.age > MAX_BEE_AGE {
            commands.entity(entity).despawn_recursive();
        }
    }

    // spawn new bee, if limit is not reached
    let Ok(build_area) = build_area.get_single() else {
        return;
    };
    let map_size = build_area
        .center
        .get_area(build_area.radius - 1)
        .size_hint()
        .0 as f32;

    if bees.iter().count() > (map_size * MAX_BEES_PER_TILE) as usize {
        return;
    }

    let mut rng = Rng::new();
    let mut elems = rng.choose_n(bee_spawning_tiles.iter(), 2);
    if elems.len() < 2 {
        return;
    }
    rng.shuffle(&mut elems);

    let (target_coord, target_type) = elems[0];
    let (origin_coord, _) = elems[1];

    let gltf = assets
        .get(&collection.bee)
        .expect("The LoadingScreen should have ensured that the asset was loaded.");
    let scene = gltf
        .scenes
        .get(BeePlugin::get_model())
        .expect("The plugin specified that the scene exists.");

    commands.spawn((
        SceneRoot(scene.clone_weak()),
        Transform::from_translation(
            Vec3::from(*origin_coord) + Vec3::new(0.1, target_type.get_tile_height() + 0.2, 0.1),
        ),
        Bee::new(*target_coord),
    ));
}

fn despawn_bee(mut commands: Commands, bees: Query<Entity, With<Bee>>) {
    for bee in bees.iter() {
        commands.entity(bee).despawn_recursive();
    }
}

/// start wing animation
fn animate_bee(
    mut commands: Commands,
    mut graphs: ResMut<Assets<AnimationGraph>>,
    tiles: Query<(&AnimationEntityLink, &Bee), Added<AnimationEntityLink>>,
    collection: Res<BeeAsset>,
    assets: Res<Assets<Gltf>>,
    mut animation_players: Query<&mut AnimationPlayer>,
) {
    let gltf = assets
        .get(&collection.bee)
        .expect("The LoadingScreen should have ensured that the asset was loaded.");
    let handle = gltf
        .animations
        .get(BeePlugin::get_animation())
        .expect("The plugin specified that the animation exists.");

    for (link, _bee) in &tiles {
        if let Ok(mut player) = animation_players.get_mut(**link) {
            let mut graph = AnimationGraph::new();
            let animation = graph.add_clip(handle.clone_weak(), 1.0, graph.root);

            player.play(animation).set_speed(2.0).repeat();
            commands
                .entity(**link)
                .insert(AnimationGraphHandle(graphs.add(graph)));
        }
    }
}

/// let bees fly around their home
fn fly_bee(time: Res<Time<Real>>, mut bees: Query<(&mut Bee, &mut Transform)>) {
    let delta = time.delta_secs();
    let elapsed: f32 = time.elapsed_secs();
    for (mut bee, mut pos) in bees.iter_mut() {
        // let the bee fly home
        let mut delta_home = Vec3::from(bee.origin)
            + Vec3::new(
                (elapsed * 4.0 + bee.age).sin(),
                0.0,
                (elapsed * 4.5 + bee.age).cos(),
            )
            - pos.translation;
        delta_home.y = 0.0;
        bee.vel = bee.vel * (1.0 - delta).clamp(0.0, 1.0)
            + delta_home.normalize_or_zero() * delta.clamp(0.0, 1.0);

        // add random movement
        bee.vel += Vec3::new(fastrand::f32() - 0.5, 0.0, fastrand::f32() - 0.5)
            * BEE_RANDOM_MOVE_FACTOR
            * delta;

        // damping speed
        bee.vel *= BEE_MOVE_DAMPING.powf(delta);

        // integrate speed
        pos.translation += bee.vel * delta;

        // rotate with noise filter
        let movement_rot = bee.vel.x.atan2(bee.vel.z) + PI / 2.0;
        let rot_delta = (movement_rot - bee.rot + PI) % (2.0 * PI) - PI;
        bee.rot += rot_delta * delta * 8.0;
        bee.rot = (bee.rot + PI) % (2.0 * PI) - PI; // cap rotation at [-pi, pi)
        pos.rotation = Quat::from_rotation_y(bee.rot);

        // age bee
        bee.age += delta;
    }
}
