use bevy::prelude::*;

use crate::{
    game::tiles::{assets::TileModels, TileType},
    settings::ActiveSettingsBank,
    AppState,
};

pub struct TileAnimations;

impl Plugin for TileAnimations {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_systems(Update, link_animations.run_if(in_state(AppState::InGame)));
        app.add_systems(
            Update,
            start_tile_animations.run_if(in_state(AppState::InGame)),
        );
        app.add_systems(
            Update,
            update_animations_setteings
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::InGame)),
        );
    }
}

#[derive(Component, Deref, DerefMut)]
pub struct AnimationEntityLink(pub Entity);

fn find_root(mut curr_entity: Entity, parent_query: &Query<&Parent>) -> Entity {
    while let Ok(parent) = parent_query.get(curr_entity) {
        curr_entity = parent.get();
    }
    curr_entity
}

/// The Animation Player is a Component of a child entity. The System links the Entity-Id to the root Entity for easy access.
pub fn link_animations(
    animation_players: Query<Entity, Added<AnimationPlayer>>,
    parents: Query<&Parent>,
    animation_link: Query<&AnimationEntityLink>,
    mut commands: Commands,
) {
    for entity in animation_players.iter() {
        let root_entity = find_root(entity, &parents);

        if animation_link.get(root_entity).is_ok() {
            warn!("Currently just one AnimationPlayer is supported!");
            continue;
        }

        commands
            .entity(root_entity)
            .insert(AnimationEntityLink(entity));
    }
}

pub fn update_animations_setteings(
    settings: Res<ActiveSettingsBank>,
    tiles: Query<&AnimationEntityLink>,
    mut animation_players: Query<&mut AnimationPlayer>,
) {
    for link in &tiles {
        if let Ok(mut player) = animation_players.get_mut(**link) {
            if settings.graphics.animations {
                player.resume_all();
            } else {
                player.pause_all();
            }
        }
    }
}

pub fn start_tile_animations(
    mut commands: Commands,
    mut graphs: ResMut<Assets<AnimationGraph>>,
    assets: Res<Assets<Gltf>>,
    models: Res<TileModels>,
    tiles: Query<(&AnimationEntityLink, &TileType), Added<AnimationEntityLink>>,
    mut animation_players: Query<&mut AnimationPlayer>,
    settings: Res<ActiveSettingsBank>,
) {
    for (link, tile) in &tiles {
        if let Some((animation, speed)) = tile.get_asset_animation() {
            if let Ok(mut player) = animation_players.get_mut(**link) {
                let handle = models
                    .0
                    .get(tile)
                    .expect("The LoadingScreen should have loaded the asset.");
                let gltf = assets
                    .get(handle)
                    .expect("The LoadingScreen should have ensured that the asset was loaded.");
                let handle = gltf
                    .animations
                    .get(animation)
                    .expect("The tile specified that the animation exists.");
                let mut graph = AnimationGraph::new();
                let animation = graph.add_clip(handle.clone_weak(), 1.0, graph.root);

                player.play(animation).set_speed(speed).repeat();
                commands
                    .entity(**link)
                    .insert(AnimationGraphHandle(graphs.add(graph)));

                if !settings.graphics.animations {
                    player.pause_all();
                }
            }
        }
    }
}
