#[cfg(feature = "graphics")]
pub mod assets;
#[cfg(feature = "graphics")]
use assets::{TileAssetsPlugin, TileModels};

pub mod chameleon_tile;
pub mod rgb_tile;

use bevy::{
    ecs::{system::EntityCommand, world::Command},
    prelude::*,
};
use chameleon_tile::{ChameleonTileData, ChameleonTilePlugin, ChameleonTileUpdate};

use crate::{coordinates::CubeCoordinate, game::hand::Deck, AppState};
#[cfg(feature = "graphics")]
use crate::{game::visuals::debug_text::DebugText, prelude::WithTranslationID, DisplayMode};
use bevy_enum_filter::prelude::*;
use serde::{Deserialize, Serialize};
use std::hash::Hash;
#[cfg(feature = "graphics")]
use std::{
    f32::consts::PI,
    hash::{DefaultHasher, Hasher},
};

#[cfg(feature = "graphics")]
use super::visuals::debug_text::DebugTextAlways;
use super::{
    map::{AdditionalTileData, Map},
    metrics::MetricsRate,
    placement_reclaim::PlacementTimer,
    resources::{BoosterTile, DefaultInfoMarker, MarketInfo, RelevantTileMarker},
};

use rgb_tile::{RgbTileData, RgbTilePlugin};

pub struct TilesPlugin {
    #[cfg(feature = "graphics")]
    pub display_mode: DisplayMode,
}

impl Plugin for TilesPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_plugins(RgbTilePlugin);
        app.add_plugins(ChameleonTilePlugin);
        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display_mode {
            app.add_plugins(TileAssetsPlugin);
        }
    }
}

/// Defines the looks and the behavior of a Tile.
#[derive(
    Component, EnumFilter, Clone, Copy, PartialEq, Eq, Hash, Debug, Serialize, Deserialize,
)]
pub enum TileType {
    Grass,
    Wheat,
    Forest,
    Windmill,
    SmallHouse,
    Beehive,
    StoneQuarry,
    StoneRocks,
    StoneHill,
    StoneMountain,
    DoubleHouse,
    Moai,
    Marketplace,
    Rgb,
    Chameleon,
}

#[cfg(feature = "graphics")]
impl WithTranslationID for TileType {
    /// Id of the name of the tiles.
    /// When added a `-description` a description about the tile should be found.
    fn get_translation_id(&self) -> &'static str {
        match self {
            Self::Grass => "tile-grass",
            Self::Wheat => "tile-wheat",
            Self::Forest => "tile-forest",
            Self::Windmill => "tile-windmill",
            Self::SmallHouse => "tile-smallhouse",
            Self::Beehive => "tile-beehive",
            Self::StoneQuarry => "tile-quarry",
            Self::StoneRocks => "tile-rocks",
            Self::StoneHill => "tile-hill",
            Self::StoneMountain => "tile-mountain",
            Self::DoubleHouse => "tile-doublehouse",
            Self::Moai => "tile-moai",
            Self::Marketplace => "tile-marketplace",
            Self::Rgb => "tile-rgb",
            Self::Chameleon => "tile-chameleon",
        }
    }
}

impl TileType {
    pub fn list() -> [Self; 15] {
        [
            Self::Grass,
            Self::Wheat,
            Self::Forest,
            Self::Windmill,
            Self::SmallHouse,
            Self::Beehive,
            Self::StoneQuarry,
            Self::StoneRocks,
            Self::StoneHill,
            Self::StoneMountain,
            Self::DoubleHouse,
            Self::Moai,
            Self::Marketplace,
            Self::Rgb,
            Self::Chameleon,
        ]
    }

    pub fn is_basic(&self) -> bool {
        !matches!(self, Self::Rgb | Self::Chameleon)
    }

    pub fn get_asset_path(&self) -> &'static str {
        match self {
            TileType::Grass => "models/grass.glb",
            TileType::Wheat => "models/building-farm.glb",
            TileType::Forest => "models/grass-forest.glb",
            TileType::Windmill => "models/building-mill.glb",
            TileType::SmallHouse => "models/building-house.glb",
            TileType::Beehive => "models/building-beehive.glb",
            TileType::StoneQuarry => "models/building-stone-quarry.glb",
            TileType::StoneRocks => "models/stone-rocks.glb",
            TileType::StoneHill => "models/stone-hill.glb",
            TileType::StoneMountain => "models/stone-mountain.glb",
            TileType::DoubleHouse => "models/building-village.glb",
            TileType::Moai => "models/building-moai.glb",
            TileType::Marketplace => "models/building-market.glb",
            TileType::Rgb => "models/rgb.glb",
            TileType::Chameleon => "models/building-farm.glb", // HACK: model will be replaced anyway. But a model needs to be supplied.
        }
    }
    pub fn get_book_link_url(&self) -> &'static str {
        match self {
            TileType::Grass => "basic/grass",
            TileType::Wheat => "basic/wheat/",
            TileType::Forest => "basic/forest",
            TileType::Windmill => "basic/windmill",
            TileType::SmallHouse => "basic/smallhouse",
            TileType::Beehive => "basic/beehive",
            TileType::StoneQuarry => "basic/quarry",
            TileType::StoneRocks => "basic/rocks",
            TileType::StoneHill => "basic/hill",
            TileType::StoneMountain => "basic/mountain",
            TileType::DoubleHouse => "basic/doublehouse",
            TileType::Moai => "basic/moai",
            TileType::Marketplace => "basic/markteplace",
            TileType::Rgb => "creative/rgb",
            TileType::Chameleon => "creative/chameleon",
        }
    }

    /// returns the scene index
    pub fn get_asset_scene(&self) -> usize {
        0
    }

    /// returns animation index and default animation speed
    pub fn get_asset_animation(&self) -> Option<(usize, f32)> {
        match self {
            TileType::Windmill => Some((0, 0.5)),
            _ => None,
        }
    }

    pub fn new_deck_in_round(round: usize) -> Deck {
        let mut deck = Deck::from([
            (Self::Grass, 2),
            (Self::Wheat, 5),
            (Self::Forest, 5),
            (Self::SmallHouse, 5),
            (Self::StoneRocks, 5),
        ]);

        if round >= 1 {
            let expansion = Deck::from([
                (Self::Beehive, 4),
                (Self::DoubleHouse, 4),
                (Self::StoneHill, 4),
                (Self::StoneQuarry, 5),
            ]);
            deck += expansion;
        }

        if round >= 2 {
            let expansion = Deck::from([(Self::Windmill, 3), (Self::StoneMountain, 3)]);
            deck += expansion;
        }

        if round >= 3 {
            let expansion = Deck::from([(Self::Moai, 2), (Self::Marketplace, 4)]);
            deck += expansion;
        }

        deck
    }

    pub fn buy_deck(round: usize) -> Deck {
        let mut deck = Deck::from([
            (Self::Grass, 0),
            (Self::Wheat, 2),
            (Self::Forest, 2),
            (Self::SmallHouse, 2),
            (Self::StoneRocks, 1),
        ]);

        if round >= 1 {
            let expansion = Deck::from([
                (Self::Beehive, 3),
                (Self::DoubleHouse, 3),
                (Self::StoneHill, 2),
                (Self::StoneQuarry, 4),
            ]);
            deck += expansion;
        }

        if round >= 2 {
            let expansion = Deck::from([(Self::Windmill, 3), (Self::StoneMountain, 3)]);
            deck += expansion;
        }

        if round >= 3 {
            let expansion = Deck::from([(Self::Moai, 2), (Self::Marketplace, 4)]);
            deck += expansion;
        }

        deck
    }

    pub fn forced_cards(round: usize) -> Vec<Self> {
        if round == 3 {
            return vec![Self::Marketplace];
        }
        Vec::new()
    }

    pub fn get_preview_path(&self) -> &'static str {
        match self {
            TileType::Grass => "previews/grass.png",
            TileType::Wheat => "previews/building-farm.png",
            TileType::Forest => "previews/grass-forest.png",
            TileType::Windmill => "previews/building-mill.png",
            TileType::SmallHouse => "previews/building-house.png",
            TileType::Beehive => "previews/building-beehive.png",
            TileType::StoneQuarry => "previews/building-stone-quarry.png",
            TileType::StoneRocks => "previews/stone-rocks.png",
            TileType::StoneHill => "previews/stone-hill.png",
            TileType::StoneMountain => "previews/stone-mountain.png",
            TileType::DoubleHouse => "previews/building-village.png",
            TileType::Moai => "previews/building-moai.png",
            TileType::Marketplace => "previews/building-market.png",
            TileType::Rgb => "previews/rgb.png",
            TileType::Chameleon => "previews/chameleon.png",
        }
    }

    #[allow(clippy::approx_constant)]
    pub fn get_tile_height(&self) -> f32 {
        match self {
            TileType::Grass => 0.318,
            TileType::Wheat => 0.4,
            TileType::Forest => 0.69,
            TileType::Windmill => 0.9,
            TileType::SmallHouse => 0.6,
            TileType::Beehive => 0.37,
            TileType::StoneQuarry => 0.76,
            TileType::StoneRocks => 0.4,
            TileType::StoneHill => 0.64,
            TileType::StoneMountain => 1.1,
            TileType::DoubleHouse => 0.6,
            TileType::Moai => 1.02,
            TileType::Marketplace => 0.6,
            TileType::Rgb => 0.2,
            TileType::Chameleon => 0.4,
        }
    }

    // returns tile orientation, or None, for random
    #[cfg(feature = "graphics")]
    fn get_rotation(&self) -> Option<TileOrientation> {
        match self {
            TileType::Windmill => Some(TileOrientation::North),
            TileType::Moai => Some(TileOrientation::North),
            TileType::Rgb => Some(TileOrientation::North),
            TileType::Chameleon => Some(TileOrientation::North),
            _ => None,
        }
    }

    /// applies custom tile data to an entity
    #[allow(clippy::single_match, unreachable_patterns)]
    fn apply_specific_data(
        &self,
        entity: &mut EntityWorldMut<'_>,
        data: Option<AdditionalTileData>,
    ) {
        match self {
            TileType::Marketplace => {
                let info: MarketInfo = data
                    .and_then(|data| match data {
                        AdditionalTileData::MarketInfo(market_info) => Some(market_info),
                        _ => None,
                    })
                    .unwrap_or_default();
                entity.insert(info);
            }
            TileType::Moai => {
                entity.insert((BoosterTile::default(), DefaultInfoMarker));
            }
            TileType::Rgb => {
                let info: RgbTileData = data
                    .and_then(|data| match data {
                        AdditionalTileData::RgbData(data) => Some(data),
                        _ => None,
                    })
                    .unwrap_or_default();
                // HACK: hides the tile until the material has been updated
                #[cfg(feature = "graphics")]
                entity.insert(Visibility::Hidden);
                entity.insert(info);
            }

            TileType::Chameleon => {
                let data = data
                    .and_then(|data| match data {
                        AdditionalTileData::ChameleonData(data) => Some(data),
                        _ => None,
                    })
                    .unwrap_or_default();

                // HACK: hides the tile until the model has been updated
                #[cfg(feature = "graphics")]
                {
                    entity.insert(Visibility::Hidden);
                    entity.insert(DebugTextAlways);
                }
                entity.insert(ChameleonTileUpdate::default());
                entity.insert(data);
            }
            _ => {
                entity.insert(DefaultInfoMarker);
            }
        };
    }

    pub fn serialize_tile_data(data: ConsistentTileData) -> Option<AdditionalTileData> {
        match data.1 {
            TileType::Marketplace => data
                .3
                .map(|data| AdditionalTileData::MarketInfo(data.clone())),
            TileType::Rgb => data.4.map(|data| AdditionalTileData::RgbData(data.clone())),
            TileType::Chameleon => data
                .5
                .map(|data| AdditionalTileData::ChameleonData(data.clone())),
            _ => None,
        }
    }
}

pub type ConsistentTileData<'a> = (
    &'a CubeCoordinate,
    &'a TileType,
    Option<&'a PlacementTimer>,
    Option<&'a MarketInfo>,
    Option<&'a RgbTileData>,
    Option<&'a ChameleonTileData>,
);

#[derive(Component, Debug, Clone, Copy, Default)]
#[allow(dead_code)]
#[cfg(feature = "graphics")]
/// Saves rotation of tiles.
enum TileOrientation {
    #[default]
    North,
    NorthEast,
    SouthEast,
    South,
    SouthWest,
    NorthWest,
}
#[cfg(feature = "graphics")]
impl From<u64> for TileOrientation {
    fn from(i: u64) -> Self {
        match i.rem_euclid(6) {
            0 => TileOrientation::North,
            1 => TileOrientation::NorthEast,
            2 => TileOrientation::SouthEast,
            3 => TileOrientation::South,
            4 => TileOrientation::SouthWest,
            5 => TileOrientation::NorthWest,
            _ => unreachable!("remainder must be smaller than 6"),
        }
    }
}
#[cfg(feature = "graphics")]
impl TileOrientation {
    /// Get orientation index.
    pub fn get_orientation(&self) -> i32 {
        match self {
            TileOrientation::North => 0,
            TileOrientation::NorthEast => 1,
            TileOrientation::SouthEast => 2,
            TileOrientation::South => 3,
            TileOrientation::SouthWest => 4,
            TileOrientation::NorthWest => 5,
        }
    }

    /// Get angle of direction in rad.
    pub fn get_angle(&self) -> f32 {
        self.get_orientation() as f32 * 60.0 / 180.0 * PI
    }
}

#[derive(Debug, Component)]
pub struct TileScene;

trait InsertTileBundle {
    fn insert_tile_data(&mut self, tile: TileType, data: Option<AdditionalTileData>) -> &mut Self;
}

impl InsertTileBundle for EntityWorldMut<'_> {
    fn insert_tile_data(&mut self, tile: TileType, data: Option<AdditionalTileData>) -> &mut Self {
        tile.apply_specific_data(self, data);
        self
    }
}

/// Descriptor of a Tile that should be added.
/// use:
/// ```ignore
/// commands.spawn_empty().add(AddTile::new(tile_type, pos));
/// ```
pub struct AddTile {
    tile_type: TileType,
    pos: CubeCoordinate,
    spawn_scene: bool,
    data: Option<AdditionalTileData>,
}

impl AddTile {
    pub fn new(tile_type: TileType, pos: impl Into<CubeCoordinate>) -> Self {
        AddTile {
            tile_type,
            pos: pos.into(),
            spawn_scene: true,
            data: None,
        }
    }

    #[allow(dead_code)]
    pub fn without_scene(mut self) -> Self {
        self.spawn_scene = false;
        self
    }
    #[allow(dead_code)]
    pub fn without_scene_cond(mut self, without_scene: bool) -> Self {
        if without_scene {
            self.spawn_scene = false;
        }
        self
    }
    #[allow(dead_code)]
    pub fn with_data(mut self, data: Option<AdditionalTileData>) -> Self {
        self.data = data;
        self
    }
}

impl From<(TileType, CubeCoordinate)> for AddTile {
    fn from(value: (TileType, CubeCoordinate)) -> Self {
        AddTile {
            tile_type: value.0,
            pos: value.1,
            spawn_scene: true,
            data: None,
        }
    }
}

impl EntityCommand for AddTile {
    fn apply(self, id: Entity, world: &mut World) {
        #[cfg(feature = "graphics")]
        let entity = if self.spawn_scene {
            let scene = {
                let models = world
                    .get_resource::<TileModels>()
                    .expect("The TileModels collection should exist at this point.");
                let model_handle = models.0.get(&self.tile_type).expect(
                    "The Loading Screen should have taken care of loading the model at this point.",
                );
                let assets = world
                    .get_resource::<Assets<Gltf>>()
                    .expect("The Gltf assets collection should exist at this point.");
                let gltf = assets
                    .get(model_handle)
                    .expect("The LoadingScreen should have ensured that the asset was loaded.");

                gltf.scenes
                    .get(self.tile_type.get_asset_scene())
                    .expect("The Scene should be available for the model")
                    .clone_weak()
            };

            let direction = self.tile_type.get_rotation().unwrap_or_else(|| {
                let mut hasher = DefaultHasher::new();
                self.pos.hash(&mut hasher);
                TileOrientation::from(hasher.finish())
            });

            let scene_bundle = (
                SceneRoot(scene),
                Transform::default().with_rotation(Quat::from_rotation_y(direction.get_angle())),
            );
            world
                .entity_mut(id)
                .insert((
                    self.tile_type,
                    self.pos,
                    Transform::from_translation(self.pos.into()),
                    Visibility::Visible,
                    direction,
                    MetricsRate::default(),
                    RelevantTileMarker::new(true),
                    DebugText::new("".into(), 0.1).with_height(self.tile_type.get_tile_height()),
                    StateScoped(AppState::InGame),
                ))
                .insert_tile_data(self.tile_type, self.data)
                .with_children(|child| {
                    child.spawn(scene_bundle).insert(TileScene);
                })
                .id()
        } else {
            world
                .entity_mut(id)
                .insert((
                    self.tile_type,
                    self.pos,
                    MetricsRate::default(),
                    RelevantTileMarker::new(false),
                    StateScoped(AppState::InGame),
                ))
                .insert_tile_data(self.tile_type, self.data)
                .id()
        };

        #[cfg(not(feature = "graphics"))]
        let entity = world
            .entity_mut(id)
            .insert((
                self.tile_type,
                self.pos,
                MetricsRate::default(),
                RelevantTileMarker::new(false),
                StateScoped(AppState::InGame),
            ))
            .insert_tile_data(self.tile_type, self.data)
            .id();

        let mut map = world
            .get_resource_mut::<Map>()
            .expect("The Map should exist at this point.");

        let tile = map.get(self.pos);

        map.set(self.pos, entity);

        // despawn old tile
        if let Some(tile) = tile {
            world.entity_mut(tile).despawn_recursive();
        }
    }
}

pub struct RemoveTile {
    pos: CubeCoordinate,
}

impl RemoveTile {
    pub fn new(pos: CubeCoordinate) -> Self {
        Self { pos }
    }
}

impl Command for RemoveTile {
    fn apply(self, world: &mut World) {
        let mut map = world
            .get_resource_mut::<Map>()
            .expect("The Map should exist at this point.");
        let tile = map.get(self.pos);

        map.remove(self.pos);

        // despawn old tile
        if let Some(tile) = tile {
            world.entity_mut(tile).despawn_recursive();
        }
    }
}
