#[cfg(feature = "graphics")]
use bevy::asset::LoadState;
use bevy::prelude::*;

use crate::AppState;
#[cfg(feature = "graphics")]
use bevy_egui::{egui, EguiContexts};
#[cfg(feature = "graphics")]
use std::collections::HashMap;

#[cfg(feature = "graphics")]
use crate::{
    errors::ErrorDisplay,
    game::asset_loading::{LACManager, LoadableAssetCollection},
    prelude::Translate,
};

#[cfg(feature = "graphics")]
use crate::DisplayMode;

use crate::game::asset_loading;
use crate::game::tiles::TileType;

#[cfg(feature = "graphics")]
pub struct TileAssetsPlugin;

#[cfg(feature = "graphics")]
impl Plugin for TileAssetsPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.register_asset_collection::<TileModels>();
        app.register_asset_collection::<TilePreviews>();
        app.add_systems(
            OnExit(AppState::LoadApp),
            finish_up_previews.run_if(in_state(DisplayMode::Graphic)),
        );
    }
}

#[cfg(feature = "graphics")]
#[derive(Resource, Default)]
pub struct TilePreviews(
    pub HashMap<TileType, Handle<Image>>,
    #[cfg(feature = "graphics")] pub HashMap<TileType, egui::TextureId>,
);

#[cfg(feature = "graphics")]
impl TilePreviews {
    pub fn get_handle(&self, tile: TileType) -> Option<Handle<Image>> {
        self.0.get(&tile).map(|h| h.clone_weak())
    }
    pub fn get_egui_handle(&self, tile: TileType) -> Option<egui::TextureId> {
        self.1.get(&tile).copied()
    }
}

#[cfg(feature = "graphics")]
impl LoadableAssetCollection for TilePreviews {
    fn load_all(asset_server: &AssetServer) -> Self {
        let mut collection = Self(HashMap::new(), HashMap::new());
        for tile in TileType::list() {
            collection
                .0
                .insert(tile, asset_server.load(tile.get_preview_path()));
        }
        collection
    }

    fn check_all(&mut self, asset_server: &AssetServer) -> asset_loading::AssetLoadStateDump {
        let mut dump = asset_loading::AssetLoadStateDump::default();

        for (_, handle) in self.0.iter() {
            dump.requested += 1;
            match asset_server.get_load_state(handle) {
                Some(LoadState::Failed(_)) => dump.failed += 1,
                Some(LoadState::Loaded) => dump.loaded += 1,
                _ => (),
            }
        }

        dump
    }
    fn get_error(
        &self,
        localization: Option<&crate::i18n::Localization>,
    ) -> Option<crate::errors::ErrorDisplay> {
        Some(ErrorDisplay {
            title: localization
                .and_then(|l| l.try_translate("missing-tile-previews-title"))
                .unwrap_or(String::from("Missing tile previews")),
            description: localization
                .and_then(|l| l.try_translate("missing-tile-previews-description"))
                .unwrap_or(String::from("Without previews, card cannot be displayed.")),
            link: None,
        })
    }
}

#[cfg(feature = "graphics")]
fn finish_up_previews(mut previews: ResMut<TilePreviews>, mut contexts: EguiContexts) {
    let mut egui_handles = HashMap::<TileType, egui::TextureId>::new();
    for (tile, handle) in previews.0.iter() {
        egui_handles.insert(*tile, contexts.add_image(handle.clone_weak()));
    }
    previews.1 = egui_handles;
}

#[derive(Resource, Default)]
#[cfg(feature = "graphics")]
pub struct TileModels(pub HashMap<TileType, Handle<Gltf>>);

#[cfg(feature = "graphics")]
impl LoadableAssetCollection for TileModels {
    fn load_all(asset_server: &AssetServer) -> Self {
        let mut collection = Self(HashMap::new());
        for tile in TileType::list() {
            collection
                .0
                .insert(tile, asset_server.load(tile.get_asset_path()));
        }
        collection
    }

    fn check_all(&mut self, asset_server: &AssetServer) -> asset_loading::AssetLoadStateDump {
        let mut dump = asset_loading::AssetLoadStateDump::default();

        for (_, handle) in self.0.iter() {
            dump.requested += 1;
            match asset_server.get_load_state(handle) {
                Some(LoadState::Failed(_)) => dump.failed += 1,
                Some(LoadState::Loaded) => dump.loaded += 1,
                _ => (),
            }
        }

        dump
    }
    fn get_error(
        &self,
        localization: Option<&crate::i18n::Localization>,
    ) -> Option<crate::errors::ErrorDisplay> {
        Some(ErrorDisplay {
            title: localization
                .and_then(|l| l.try_translate("missing-tile-models-title"))
                .unwrap_or(String::from("Missing tile models")),
            description: localization
                .and_then(|l| l.try_translate("missing-tile-models-description"))
                .unwrap_or(String::from(
                    "Without the 3D models, tiles won't be visible.",
                )),
            link: None,
        })
    }
}
