use std::{cmp, time::Duration};

#[cfg(feature = "graphics")]
use bevy::time::common_conditions::on_timer;

use bevy::{prelude::*, state::state::FreelyMutableState};
use serde::{Deserialize, Serialize};

use crate::{
    game::gamemodes::{
        campaign::levels::{
            challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
        },
        challenge::ChallengeGameState,
        zen::ZenGameState,
    },
    AppState, DisplayMode,
};

use super::{
    controls::PlacerSet,
    hand::{fill_hand, Deck, Hand, RandomCore},
    metrics::{Metrics, MetricsRate, TargetMetrics},
    report::Warnings,
    rewards::rewards_timer_end,
    tiles::TileType,
    time::Game,
    GameEnd, GamePauseState, GameSimSet,
};

#[cfg(feature = "graphics")]
use super::metrics::reset_metrics;
#[cfg(feature = "graphics")]
use crate::game::gamemodes::{campaign::levels::CampaignGroups, GameMode};

pub const ROUND_DURATION: Duration = Duration::from_secs(60);
pub const NEW_HAND_INTERVAL: Duration = Duration::from_secs(20);

macro_rules! define_round_plugin {
    ($({
        // enum name, containing both state and between
        states: $states:ident,
        // in which state to run the round processor
        state: $state:expr,
        // state when between rounds
        between: $between:expr,
        // gamemode to handle
        gamemode: $gm:expr,
        // system that checks if the round is over and performs
        round_end_processor: $rep:ident,
    }),*) => {
        pub struct RoundPlugin {
            #[allow(dead_code)]
            pub display_mode: DisplayMode,
        }
        impl Plugin for RoundPlugin {
            fn build(&self, app: &mut bevy::prelude::App) {
                app.init_resource::<NewHandTimer>();
                app.init_resource::<ManualRedrawCounter>();
                app.init_resource::<RoundTimer>();
                app.init_resource::<RoundCounter>();
                app.init_resource::<HandRedrawCost>();
                app.init_resource::<MetricsHistory>();
                app.add_event::<ManualRedrawEvent>();

                use crate::prelude::never;

                app.add_systems(
                    Update,
                    round_processor
                        .run_if(never()$(.or(in_state($state)))*),
                );

                $(
                    app.add_systems(
                        Update,
                        $rep::<$states>
                            .in_set(GameSimSet::AfterSim)
                            .after(round_processor)
                            .run_if(in_state($state)),
                    );
                )*
                app.add_systems(
                    Update,
                    update_redraw_cost
                        .in_set(GameSimSet::AfterSim)
                        .after(rewards_timer_end)
                        .after(round_processor)
                        .run_if(never()$(.or(in_state($state)))*),
                );
                $(
                    app.add_systems(
                        OnEnter($state),
                        update_redraw_cost.after(continue_timer),
                    );
                    app.add_systems(OnEnter($state), continue_timer);
                )*

                app.add_systems(OnEnter(AppState::InGame), reset_round_resources);

                app.add_systems(
                    Update,
                    process_manual_redraw
                        .in_set(GameSimSet::Prepare)
                        .run_if(never()$(.or(in_state($state)))*)
                        .after(PlacerSet::Placer)
                        .after(rewards_timer_end),
                );

                app.add_event::<PhaseEnd>();
                app.add_event::<FreeRedraw>();
                app.add_event::<Redraw>();
                app.add_event::<GameOver>();

                if let DisplayMode::Headless = self.display_mode {
                    $(
                        app.add_systems(
                            Update,
                            continue_next_round::<$states>
                                .run_if(in_state($between)),
                        );
                    )*
                }

                #[cfg(feature = "graphics")]
                if let DisplayMode::Graphic = self.display_mode {
                    app.init_resource::<StatisticsData>();
                    app.add_systems(
                        Update,
                        track_stats_timed
                            .run_if(in_state(GamePauseState::Running))
                        .run_if(never()$(.or(in_state($state)))*),
                    );
                    // collect data every round
                    app.add_systems(
                        Update,
                        collect_stats
                            .run_if(on_timer(ROUND_DURATION))
                        .run_if(never()$(.or(in_state($state)))*),
                    );
                    $(
                        // collect data on round startup
                        // and reset data from possible past round
                        app.add_systems(
                            OnEnter($gm),
                            (reset_stats_tracker, collect_stats.after(reset_metrics)).chain(),
                        );
                        app.add_systems(OnExit($between), collect_stats);
                        app.add_systems(OnEnter($between), collect_stats);
                    )*
                }
            }
        }
    }
}

define_round_plugin![
    {
        states: ChallengeGameState,
        state: ChallengeGameState::InRound,
        between: ChallengeGameState::BetweenRounds,
        gamemode: GameMode::Challenge,
        round_end_processor: round_end_processor,
    },
    {
        states: CampaignChallengeState,
        state: CampaignChallengeState::InRound,
        between: CampaignChallengeState::BetweenRounds,
        gamemode: CampaignGroups::CampaignChallenge,
        round_end_processor: round_end_processor,
    },
    {
        states: CampaignWettbewerbState,
        state: CampaignWettbewerbState::InRound,
        between: CampaignWettbewerbState::BetweenRounds,
        gamemode: CampaignGroups::CampaignWettbewerb,
        round_end_processor: round_end_processor,
    },
    {
        states: ZenGameState,
        state: ZenGameState::InRound,
        between: ZenGameState::BetweenRounds,
        gamemode: GameMode::Zen,
        round_end_processor: zen_round_end_processor,
    }
];

#[derive(Debug, Clone, Resource, Deref, DerefMut, Default, Serialize, Deserialize)]
pub struct RoundCounter(pub usize);
#[derive(Resource, Deref, DerefMut, Default)]
pub struct ManualRedrawCounter(pub usize);

#[derive(Event)]
pub struct FreeRedraw;
#[derive(Event)]
pub struct Redraw;
#[derive(Event)]
pub struct GameOver;
#[derive(Event)]
pub struct PhaseEnd;

/// price factor per card still in hand
/// multiplied by the actual cost
/// values >1 make the price go up
const CARD_MARKUP: f32 = 1.2;
/// price factor for getting new cards early
/// multiplied by the actual cost
/// values >1 make the price go up
const TIME_MARKUP: f32 = 1.1;
/// price factor for redrawing a lot
/// multiplied by the actual cost
/// values >1 make the price go up
const REDRAW_MARKUP: f32 = 1.4;
/// base price factor
/// multiplied by target_metrics
const BASE_MARKUP: f32 = 0.05;

/// The cost to redraw the hand now
/// if the option is None the redraw is free
/// otherwise the price per metric is he option value
#[derive(Resource, Deref, DerefMut, Default, Clone, Serialize, Deserialize, Debug)]
pub struct HandRedrawCost(Option<Metrics>);
/// precalculate the redraw price
/// should run on HandTimer and RedrawCounter change
/// and when the hand or target metrics change
pub fn update_redraw_cost(
    hand_interval: Res<NewHandTimer>,
    mut cost: ResMut<HandRedrawCost>,
    counter: Res<ManualRedrawCounter>,
    hand: Res<Hand>,
    target_metrics: Res<TargetMetrics>,
    mut free_redraw: EventWriter<FreeRedraw>,
) {
    // if the cooldown is over, redrawing the cards is free
    if hand_interval.finished() {
        if cost.is_some() {
            free_redraw.send(FreeRedraw);
        }
        **cost = None;
        return;
    }
    let fraction = if hand_interval.duration() == Duration::ZERO {
        0.0
    } else {
        1.0 - hand_interval.elapsed().as_secs() as f32 / hand_interval.duration().as_secs() as f32
    };
    let markup = REDRAW_MARKUP * (1 + counter.0) as f32
        + TIME_MARKUP * fraction
        + CARD_MARKUP * hand.items.len() as f32;

    **cost = Some(Metrics {
        materials: target_metrics.materials * BASE_MARKUP * markup,
        money: target_metrics.money * BASE_MARKUP * markup,
        food: target_metrics.food * BASE_MARKUP * markup,
    })
}

#[derive(Resource, Deref, DerefMut)]
pub struct RoundTimer(pub Timer);
impl Default for RoundTimer {
    fn default() -> Self {
        Self(Timer::new(ROUND_DURATION, TimerMode::Once))
    }
}

#[derive(Resource, Deref, DerefMut)]
pub struct NewHandTimer(pub Timer);

impl Default for NewHandTimer {
    fn default() -> Self {
        Self(Timer::new(NEW_HAND_INTERVAL, TimerMode::Once))
    }
}

#[derive(Event)]
pub struct ManualRedrawEvent;

#[allow(clippy::too_many_arguments)]
pub fn process_manual_redraw(
    mut ev: EventReader<ManualRedrawEvent>,
    mut counter: ResMut<ManualRedrawCounter>,
    mut hand_interval: ResMut<NewHandTimer>,
    rng: ResMut<RandomCore>,
    hand: ResMut<Hand>,
    deck: ResMut<Deck>,
    mut metrics: ResMut<Metrics>,
    redraw_cost: Res<HandRedrawCost>,
    round_counter: Res<RoundCounter>,
    mut redraw: EventWriter<Redraw>,
    mut warnings: EventWriter<Warnings>,
) {
    if ev.is_empty() {
        return;
    }
    ev.clear();

    let (rng, hand, deck) = (rng.into_inner(), hand.into_inner(), deck.into_inner());

    if let Some(price) = &redraw_cost.0 {
        // player has to pay the price to redraw cards now
        if &*metrics >= price {
            // player can afford the price
            *metrics -= price;
            // increase manual redraw counter
            **counter += 1;
            *deck += TileType::buy_deck(round_counter.0);
            hand.reset();
        } else {
            warnings.send(Warnings::redraw());
            return;
        }
    };
    fill_hand(rng, hand, deck);

    // reselect hand slot
    if hand.selected.is_none() {
        hand.selected = Some(0);
    }

    // restart timer
    hand_interval.reset();
    hand_interval.unpause();
    redraw.send(Redraw);
}

#[allow(clippy::extra_unused_type_parameters)]
fn round_processor(
    time: Res<Time<Game>>,
    mut round_timer: ResMut<RoundTimer>,
    mut hand_interval: ResMut<NewHandTimer>,
) {
    // advance the timer
    // won't to anything if the timer is paused
    round_timer.tick(time.delta());
    hand_interval.tick(time.delta());
}

/// if the player missed the metrics target this many times,
/// the game is considered lost
pub const METRICS_BOX_TRACKING_SIZE: usize = 2;

/// the vec stores the metrics of the last rounds
/// [first] [second] ... [last]
#[derive(Resource, Deref, DerefMut)]
pub struct MetricsHistory(pub Vec<(Metrics, TargetMetrics)>);

impl Default for MetricsHistory {
    fn default() -> Self {
        Self(Vec::with_capacity(METRICS_BOX_TRACKING_SIZE))
    }
}

/// Supporttrait so that we can get the correct State values from GameMode states
pub trait RoundSystemSupport: States {
    /// returns the state, to be opened if the player lost the game
    fn get_gameover_state() -> Self;
    /// returns the state, to be opened if the current round is over
    fn get_phaseend_state() -> Self;
    /// returns the state that represents a ongoing round
    fn get_inround_state() -> Self;
    /// returns the read-only interaction state
    /// if the gamemode supports spectate mode
    fn get_spectate_state() -> Option<Self>;
}

/// default challenge round end processor
/// ends the round after a timer has expired
/// and checks the metrics using a box/window tracker
/// to control the next game state
#[allow(clippy::too_many_arguments)]
pub fn round_end_processor<STATES: SubStates + RoundSystemSupport + FreelyMutableState>(
    mut round_timer: ResMut<RoundTimer>,
    mut hand_interval: ResMut<NewHandTimer>,
    metrics: Res<Metrics>,
    target_metrics: Res<TargetMetrics>,
    mut metrics_history: ResMut<MetricsHistory>,
    mut round_counter: ResMut<RoundCounter>,
    mut phaseend: EventWriter<PhaseEnd>,
    mut gameover: EventWriter<GameOver>,
    mut gameend: EventWriter<GameEnd>,
    mut next_state: ResMut<NextState<STATES>>,
    mut next_pause: ResMut<NextState<GamePauseState>>,
) {
    if round_timer.just_finished() {
        // pause the timer
        // should be reactivated by an UI element that says "Next round"
        round_timer.pause();
        hand_interval.pause();

        // remove old history data
        for i in 0..metrics_history.len() as isize - METRICS_BOX_TRACKING_SIZE as isize {
            metrics_history.remove(i as usize);
        }

        // append metrics information from last round
        metrics_history.push((
            metrics.into_inner().clone(),
            target_metrics.into_inner().clone(),
        ));

        // process metrics
        let mut missed_target: usize = 0;
        for i in 0..cmp::min(metrics_history.len(), METRICS_BOX_TRACKING_SIZE) {
            if let Some((metrics, target)) = metrics_history.get(metrics_history.len() - 1 - i) {
                if metrics.food < target.food
                    || metrics.materials < target.materials
                    || metrics.money < target.money
                {
                    missed_target += 1;
                }
            }
        }

        // increase round counter
        // has the welcoming sideeffect, that the Non-Round screens
        // can simply display the round number without adding 1
        round_counter.0 += 1;

        if missed_target < METRICS_BOX_TRACKING_SIZE {
            // player reached target often enough
            // change game state to between rounds
            // the game should display a dialog that allows the user to continue to the next round
            // (or automatically go to the next round in headless mode)
            next_state.set(STATES::get_phaseend_state());
            phaseend.send(PhaseEnd);
        } else {
            // player lost the game
            // as they missed the target too often
            next_state.set(STATES::get_gameover_state());
            gameover.send(GameOver);
            gameend.send(GameEnd);
        }
        next_pause.set(GamePauseState::Paused);
    }
}

/// round end processor used in zen mode
/// the next round is always reached once all resource targets were met
/// meaning that the gameover state cannot be reached
#[allow(clippy::too_many_arguments)]
pub fn zen_round_end_processor<STATES: SubStates + RoundSystemSupport + FreelyMutableState>(
    mut round_timer: ResMut<RoundTimer>,
    mut hand_interval: ResMut<NewHandTimer>,
    metrics: Res<Metrics>,
    target_metrics: Res<TargetMetrics>,
    mut round_counter: ResMut<RoundCounter>,
    mut phaseend: EventWriter<PhaseEnd>,
    mut next_state: ResMut<NextState<STATES>>,
    mut next_pause: ResMut<NextState<GamePauseState>>,
) {
    if metrics.food >= target_metrics.food
        && metrics.materials >= target_metrics.materials
        && metrics.money >= target_metrics.money
    {
        // pause the timer
        // should be reactivated by an UI element that says "Next round"
        hand_interval.pause();
        // NOTE: the zen timer is used for virtual deck refills
        // whenever the timer finishes, this should happen whenever the normal
        // challenge round would have been over, so we have to pause the timer
        // exactly the way we would do normally
        round_timer.pause();

        // increase round counter
        // has the welcoming sideeffect, that the Non-Round screens
        // can simply display the round number without adding 1
        round_counter.0 += 1;

        // change game state to between rounds
        // the game should display a dialog that allows the user to continue to the next round
        // (or automatically go to the next round in headless mode)
        next_state.set(STATES::get_phaseend_state());
        phaseend.send(PhaseEnd);
        next_pause.set(GamePauseState::Paused);
    }
}

fn continue_next_round<STATES: SubStates + RoundSystemSupport + FreelyMutableState>(
    mut next_state: ResMut<NextState<STATES>>,
    mut next_pause: ResMut<NextState<GamePauseState>>,
) {
    next_state.set(STATES::get_inround_state());
    next_pause.set(GamePauseState::Running);
}

pub fn continue_timer(
    mut round_timer: ResMut<RoundTimer>,
    mut hand_interval: ResMut<NewHandTimer>,
    mut manual_redraw_counter: ResMut<ManualRedrawCounter>,
    mut next_pause: ResMut<NextState<GamePauseState>>,
) {
    round_timer.reset();
    round_timer.unpause();

    hand_interval.reset();
    hand_interval.unpause();

    // reset the manual redraw counter to 0 at the beginning of each round
    manual_redraw_counter.0 = 0;

    next_pause.set(GamePauseState::Running);
}

/// resets resources at start of game
pub fn reset_round_resources(
    mut round_timer: ResMut<RoundTimer>,
    mut hand_interval: ResMut<NewHandTimer>,
    mut history: ResMut<MetricsHistory>,
    mut round_counter: ResMut<RoundCounter>,
    mut manual_redraw_counter: ResMut<ManualRedrawCounter>,
) {
    round_timer.reset();
    round_timer.unpause();
    hand_interval.reset();
    hand_interval.unpause();
    *history = MetricsHistory::default();
    round_counter.0 = 0;
    manual_redraw_counter.0 = 0;
}

/// how many samples to save per round
#[cfg(feature = "graphics")]
const DATA_POINTS_PER_ROUND: u32 = 5;

#[derive(Default, Resource)]
pub struct StatisticsData {
    /// collect metrics data
    /// because Metrics and TargetMetrics are supposed to be displayed in the same plot
    /// we might as well store them in the same vec
    pub metrics: Vec<(Duration, Metrics, TargetMetrics)>,
    /// collect metrics rate
    /// should be displayed in a separate graph
    pub metrics_rate: Vec<(Duration, MetricsRate)>,
    /// point in time when we stored the last data point
    #[allow(dead_code)]
    last_data: Duration,
}

/// restore the default values for every attribute of the statistics data
#[cfg(feature = "graphics")]
fn reset_stats_tracker(mut data: ResMut<StatisticsData>) {
    data.metrics = Default::default();
    data.metrics_rate = Default::default();
    data.last_data = Default::default();
}

/// stores metrics data a specific number of times per round
/// because the [`StatisticsTime`] only runs in-game
/// the ellapsed time should be equal to the time spent in a round
#[cfg(feature = "graphics")]
fn track_stats_timed(
    time: Res<Time<Game>>,
    metrics: Res<Metrics>,
    target_metrics: Res<TargetMetrics>,
    metrics_rate: Res<MetricsRate>,
    mut data: ResMut<StatisticsData>,
) {
    let now = time.elapsed();

    if now - data.last_data >= ROUND_DURATION / DATA_POINTS_PER_ROUND {
        // collect new data set
        data.metrics
            .push((now, metrics.clone(), target_metrics.clone()));
        data.metrics_rate.push((now, metrics_rate.clone()));
        // store the current time
        data.last_data = now;
    }
}

/// simply collects the current metrics data
/// schedule this at the exact time when you want to run it
#[cfg(feature = "graphics")]
fn collect_stats(
    time: Res<Time<Game>>,
    metrics: Res<Metrics>,
    target_metrics: Res<TargetMetrics>,
    metrics_rate: Res<MetricsRate>,
    mut data: ResMut<StatisticsData>,
) {
    let now = time.elapsed();
    data.metrics
        .push((now, metrics.clone(), target_metrics.clone()));
    data.metrics_rate.push((now, metrics_rate.clone()));
}
