use std::{collections::HashSet, time::Duration};

use bevy::{app::Plugin, prelude::*};

use fastrand::Rng;
use serde::{Deserialize, Serialize};

use crate::{
    coordinates::CubeCoordinate,
    game::{
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
                CampaignGroups,
            },
            challenge::ChallengeGameState,
            zen::ZenGameState,
            GameMode,
        },
        report::Warnings,
    },
    random::TTERand,
    DisplayMode,
};

use super::{
    controls::PlacerSet,
    hand::RandomCore,
    metrics::{Metrics, MetricsRate},
    resources::marketplace::re_collect_resource_rates,
    tiles::TileType,
    time::Game,
    GameSimSet,
};

const REWARD_INTERVAL: Duration = Duration::from_millis(19_000);
const MAX_REWARDS: usize = 2;

/// Secounds worth of current production rate given to the player.
const REWARD_FACTOR_MIN: f32 = 1.0;
const REWARD_FACTOR_MAX: f32 = 3.0;

macro_rules! define_rewards_plugin {
    ($({
        // state in which rewards should be generated
        state: $state:expr,
        // gamemode to handle
        gamemode: $gm:expr$(,)?
    }),*) => {
        pub struct RewardsPlugin {
            #[allow(dead_code)]
            pub display_mode: DisplayMode,
        }
        impl Plugin for RewardsPlugin {
            fn build(&self, app: &mut bevy::prelude::App) {
                app.init_resource::<RewardsTimer>();
                app.add_event::<RewardReqEvent>();
                app.add_event::<RewardEvent>();

                use crate::prelude::never;

                app.add_systems(
                    Update,
                    (
                        rewards_timer_end,
                        handle_rewards_click.after(re_collect_resource_rates),
                    )
                        .after(PlacerSet::Placer)
                        .in_set(GameSimSet::Prepare)
                        .run_if(never()$(.or(in_state($state)))*),
                );

                $(
                    app.add_systems(OnEnter($gm), reset_rewards_timer);
                    app.add_systems(OnExit($gm), despawn_rewards);
                )*
            }
        }
    }
}

define_rewards_plugin![
    {
        state: ChallengeGameState::InRound,
        gamemode: GameMode::Challenge
    },
    {
        state: CampaignChallengeState::InRound,
        gamemode: CampaignGroups::CampaignChallenge
    },
    {
        state: CampaignWettbewerbState::InRound,
        gamemode: CampaignGroups::CampaignWettbewerb
    },
    {
        state: ZenGameState::InRound,
        gamemode: GameMode::Zen
    }
];

#[derive(Resource, Deref, DerefMut)]
pub struct RewardsTimer(pub Timer);
impl Default for RewardsTimer {
    fn default() -> Self {
        Self(Timer::new(REWARD_INTERVAL, TimerMode::Once))
    }
}

pub fn rewards_timer_end(
    time: Res<Time<Game>>,
    mut rewards_timer: ResMut<RewardsTimer>,
    mut commands: Commands,
    rewards: Query<&CubeCoordinate, With<Reward>>,
    tiles: Query<(&CubeCoordinate, &TileType), With<TileType>>,
    mut rng: ResMut<RandomCore>,
) {
    rewards_timer.tick(time.delta());

    if !rewards.is_empty() {
        rewards_timer.reset();
        return;
    }
    if rewards_timer.just_finished() {
        rewards_timer.reset();
        rewards_timer.unpause();

        let rewards: HashSet<CubeCoordinate> =
            HashSet::from_iter(rewards.iter().copied().take_while(|_| true));

        // choose positions to spawn rewards
        for (coord, tile_type) in rng.choose_n(
            tiles
                .iter()
                .sort_by::<&CubeCoordinate>(|a, b| a.cmp(b))
                .filter(|&x| !rewards.contains(x.0)),
            MAX_REWARDS,
        ) {
            let factor = rng.f32() * (REWARD_FACTOR_MAX - REWARD_FACTOR_MIN) + REWARD_FACTOR_MIN;
            commands.spawn((
                Reward::new(
                    RewardType::get_random(&mut rng),
                    20.0,
                    factor,
                    (factor - REWARD_FACTOR_MIN) / (REWARD_FACTOR_MAX - REWARD_FACTOR_MIN),
                    tile_type.get_tile_height(),
                ),
                *coord,
            ));
        }
    }
}

fn despawn_rewards(rewards: Query<Entity, With<Reward>>, mut commands: Commands) {
    for entity in rewards.iter() {
        commands.entity(entity).despawn_recursive();
    }
}

pub fn reset_rewards_timer(mut bouns_timer: ResMut<RewardsTimer>) {
    bouns_timer.reset();
    bouns_timer.unpause();
}

#[derive(Clone, Copy, Debug, Serialize, Deserialize)]
pub enum RewardType {
    Food,
    Materials,
    Money,
}

impl RewardType {
    pub fn list() -> Vec<Self> {
        vec![Self::Food, Self::Materials, Self::Money]
    }
    fn get_random(rng: &mut Rng) -> Self {
        let paths = RewardType::list();
        let index = rng.u32(0..paths.len() as u32) as usize;
        paths[index]
    }
}

#[derive(Component, Clone, Copy, Debug, Serialize, Deserialize)]
pub struct Reward {
    pub kind: RewardType,
    #[serde(skip)]
    pub metrics_rate_factor: f32,
    #[serde(skip)]
    pub amount: f32,
    pub val: f32,
    #[serde(skip)]
    pub base_height: f32,
}
#[derive(Component, Clone, Copy, Debug, Serialize, Deserialize)]
pub struct RewardSend {
    #[serde(flatten)]
    pub reward: Reward,
    pub coord: CubeCoordinate,
}

#[derive(Event, Clone, Copy, Debug, Deref, DerefMut)]
pub struct RewardReqEvent(pub CubeCoordinate);

#[derive(Event, Clone, Copy, Debug, Deref, DerefMut)]
pub struct RewardEvent(pub CubeCoordinate);

impl Reward {
    /// Create new reward.
    /// `val` is value in 0.0..1.0 and used for asset selection.
    fn new(
        kind: RewardType,
        amount: f32,
        metrics_rate_factor: f32,
        val: f32,
        base_height: f32,
    ) -> Self {
        Self {
            kind,
            amount,
            metrics_rate_factor,
            val,
            base_height,
        }
    }
}

#[allow(clippy::extra_unused_type_parameters)]
pub fn handle_rewards_click(
    mut commands: Commands,
    mut rewards_event: EventReader<RewardReqEvent>,
    mut rewards_succ_event: EventWriter<RewardEvent>,
    mut metrics: ResMut<Metrics>,
    metrics_rate: Res<MetricsRate>,
    rewards: Query<(Entity, &Reward, &CubeCoordinate), With<Reward>>,
    mut warnings: EventWriter<Warnings>,
) {
    let mut handled: HashSet<Entity> = HashSet::new();
    for pos in rewards_event.read() {
        let Some((entity, reward_data, _)) = rewards.iter().find(|r| *r.2 == **pos) else {
            warn!(
                "Tried collecting reward from {:?}, but there is no reward!",
                **pos
            );
            warnings.send(Warnings::reward());
            continue;
        };
        if handled.contains(&entity) {
            continue;
        }
        handled.insert(entity);

        match reward_data.kind {
            RewardType::Food => {
                metrics.food +=
                    reward_data.amount + metrics_rate.food * reward_data.metrics_rate_factor
            }
            RewardType::Materials => {
                metrics.materials +=
                    reward_data.amount + metrics_rate.materials * reward_data.metrics_rate_factor
            }
            RewardType::Money => {
                metrics.money +=
                    reward_data.amount + metrics_rate.money * reward_data.metrics_rate_factor
            }
        }
        rewards_succ_event.send(RewardEvent(**pos));
        commands.entity(entity).despawn_recursive();
    }
}
