use std::collections::{HashMap, HashSet};

use bevy::prelude::*;
use bevy_enum_filter::Enum;

use crate::{
    coordinates::CubeCoordinate,
    game::{map::Map, metrics::MetricsRate, tiles::tile_type_filters},
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

use super::{RelKind, RelevantTileMarker};

/// Radius in which forest tiles have negative effect on windmills. Tweak this for game balancing
const FOREST_RADIUS: usize = 1;
/// Lost production efficiency due to forests in FOREST_RADIUS (additive). Tweak this for game balancing
const EFFICIENCY_LOSS_PER_FOREST: f32 = 0.5;
/// Radius in which wheat tiles are being harvested (and processed) by mills. Tweak this for game balancing
const WHEAT_RADIUS: usize = 3;
/// How much wheat a wheat tile supplies to be harvested by mills in WHEAT_RADIUS. Tweak this for game balancing
const WHEAT_SUPPLY: f32 = 2.0;
/// Maximum amount a mill can harvest from one wheat-tile. Tweak this for game balancing
const MILL_HARVEST_POWER: f32 = 1.0;
/// Final multiplier for mill production. Tweak this for game balancing
const MILL_REFINERY_MULTIPLIER: f32 = 2.0;

pub fn calc_windmill_tile(
    map: Res<Map>,
    #[cfg(feature = "graphics")] mut debug_text: Query<&mut DebugText>,
    mut mill: Query<
        (
            Entity,
            &CubeCoordinate,
            &mut MetricsRate,
            &mut RelevantTileMarker,
        ),
        With<Enum!(TileType::Windmill)>,
    >,
    wheat: Query<&CubeCoordinate, With<Enum!(TileType::Wheat)>>,
    forest: Query<&CubeCoordinate, With<Enum!(TileType::Forest)>>,
) {
    // needed to look up mills by CubeCoordinates
    let mut mills: HashSet<CubeCoordinate> = HashSet::with_capacity(map.len() / 2);
    for (_, &i, _, _) in mill.iter() {
        mills.insert(i);
    }

    // HashMap<cords of wheat tile, harvest for its mills in range>
    let mut wheat_reg: HashMap<CubeCoordinate, f32> = HashMap::with_capacity(map.len() / 2);
    for &wht in wheat.iter() {
        // find number of windmills in range
        let mills_in_radius = wht
            .get_area(WHEAT_RADIUS)
            .filter(|x| mills.contains(x))
            .count();

        // Insert harvest capacity per mill. Take the minimum of: one mill is harvesting vs. multiple mills.
        wheat_reg.insert(
            wht,
            MILL_HARVEST_POWER.min(WHEAT_SUPPLY / mills_in_radius as f32),
        );
    }

    for (_entity, coord, mut rate, mut relevant) in mill.iter_mut() {
        relevant.reset();
        // find number of forests in area
        let forest_neighbors = map
            .get_area_with_coords(*coord, FOREST_RADIUS)
            .filter(|(entity, coord)| {
                if forest.contains(*entity) {
                    relevant.add(*coord, RelKind::Secondary);
                    true
                } else {
                    false
                }
            })
            .count();

        // get total harvest of this mill
        let mut harvest_sum = 0.;
        for i in coord.get_area(WHEAT_RADIUS) {
            if let Some(&harvest) = wheat_reg.get(&i) {
                relevant.add(i, RelKind::Primary);
                harvest_sum += harvest;
            }
        }
        // lost efficiency due to close forest tiles
        let efficiency = 1.0 - (forest_neighbors as f32 * EFFICIENCY_LOSS_PER_FOREST);
        let profit = MILL_REFINERY_MULTIPLIER * efficiency * harvest_sum;
        *rate = MetricsRate {
            food: profit,
            materials: 0.0,
            money: 0.0,
        };

        #[cfg(feature = "graphics")]
        if let Ok(mut t) = debug_text.get_mut(_entity) {
            t.add_section(
                "Windmill:",
                format!(
                    "Blocking Forest {}\nproduces: {:.2}",
                    forest_neighbors, profit
                ),
            );
        }
    }
}
