use std::{collections::HashSet, f32::consts::E};

use bevy::prelude::*;
use bevy_enum_filter::Enum;

use crate::{
    coordinates::CubeCoordinate,
    game::{map::Map, metrics::MetricsRate, resources::RelKind, tiles::tile_type_filters},
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

use super::RelevantTileMarker;

/// if group size == OPTIMAL_GROUP_SIZE then profit_per_node = OPTIMAL_PRODUCTION_RATE.
///
/// The closer group size to OPTIMAL_GROUP_SIZE the better.
pub const OPTIMAL_GROUP_SIZE: f32 = 9.;
/// Production of wheat tiles if group size == OPTIMAL_GROUP_SIZE. Tweak this for game balancing
pub const OPTIMAL_PRODUCTION_RATE: f32 = 3.4;
/// if group size >= MAXIMUM_GROUP_SIZE then profit_per_node = BASE_PRODUCTION_RATE. Tweak this for game balancing
pub const MAXIMUM_GROUP_SIZE: f32 = 5. + OPTIMAL_GROUP_SIZE;
/// Base Production of wheat tile (group size == 1 or group size >= MAXIMUM_GROUP_SIZE). Tweak this for game balancing
pub const BASE_PRODUCTION_RATE: f32 = 0.1;

pub fn calc_wheat_tile(
    map: Res<Map>,
    #[cfg(feature = "graphics")] mut debug_text: Query<
        &mut DebugText,
        With<Enum!(TileType::Wheat)>,
    >,
    mut tile_data: Query<(&mut MetricsRate, &mut RelevantTileMarker), With<Enum!(TileType::Wheat)>>,
    wheat: Query<&CubeCoordinate, With<Enum!(TileType::Wheat)>>,
) {
    let mut used: HashSet<CubeCoordinate> = HashSet::new();

    fn profit(group_size: usize) -> f32 {
        if group_size == 1 {
            return BASE_PRODUCTION_RATE;
        }

        let x = group_size as f32;

        /// gaussian-function
        fn gaussian(x: f32) -> f32 {
            E.powf(-x * x)
        }

        // Width of the distance between relevant points: [1, OPT] or (OPT, MAX] divided by 2.
        let width = if x <= OPTIMAL_GROUP_SIZE {
            (OPTIMAL_GROUP_SIZE - 1.) / 2.0
        } else {
            (OPTIMAL_GROUP_SIZE - MAXIMUM_GROUP_SIZE) / 2.0
        };

        // apply bell curve to x, but stretched to fit dimensions.
        gaussian((x - OPTIMAL_GROUP_SIZE) / width)
            * (OPTIMAL_PRODUCTION_RATE - BASE_PRODUCTION_RATE)
            + BASE_PRODUCTION_RATE
    }

    #[allow(unused_variables)]
    let mut group_id = 0;

    for coord in wheat.iter() {
        if used.contains(coord) {
            continue;
        }
        let group: HashSet<CubeCoordinate> =
            map.depth_search_collect(*coord, |_, e| wheat.contains(e));
        let group_size = group.len();
        // let profit_per_node = profit(group_size as f32);
        let profit_per_node = profit(group_size);

        for c in group.iter() {
            if let Some(Ok((mut rate, mut relevant))) = map.get(*c).map(|e| tile_data.get_mut(e)) {
                relevant.reset();
                for gcoord in group.iter() {
                    if gcoord != c {
                        relevant.add(*gcoord, RelKind::Primary);
                    }
                }
                *rate = MetricsRate {
                    food: profit_per_node,
                    materials: 0.0,
                    money: 0.0,
                };
            }
            #[cfg(feature = "graphics")]
            if let Some(Ok(mut t)) = map.get(*c).map(|e| debug_text.get_mut(e)) {
                t.add_section(
                    "Wheat Group:",
                    format!(
                        "G-ID: {}  --  size: {}\nproduces: {:.2}",
                        group_id, group_size, profit_per_node
                    ),
                );
            }
        }

        used.extend(group);
        group_id += 1;
    }
}
