use std::collections::{HashMap, HashSet};

use bevy::prelude::*;
use bevy_enum_filter::Enum;

use crate::{
    coordinates::CubeCoordinate,
    game::{map::Map, metrics::MetricsRate, tiles::tile_type_filters},
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

use super::{RelKind, RelevantTileMarker};

/// Helper data structure to safe from which stoney-tile a quarry is mining form.
///
/// Saves the rock_level of the stoney-tile for convenience.
/// r is 0 if s is None meaning the quarry is not mining from a stoney-tile.
#[derive(PartialEq, Eq, Debug, Copy, Clone)]
struct QuarryStoney {
    /// quarry
    q: CubeCoordinate,
    /// the stoney-tile that q is mining
    s: Option<CubeCoordinate>,
    /// rocklevel
    r: u8,
}

/// Helper data structure for registrations list in solve_via_registration_list
#[derive(Debug)]
struct WantMe {
    /// quarries that want to mine this stoney tile
    list: Vec<CubeCoordinate>,
    /// free slots left if all quarries in list were to mine this stoney tile
    num_free_slots: i8,
}

/// Struct to hold a current solution.
#[derive(Clone)]
struct Solution {
    /// Production rate (for convenience)
    production: usize,
    /// collection for which quarry uses which stoney tile
    output: Vec<QuarryStoney>,
    /// stoney-tile-coords, (free, max)
    stoney_output: HashMap<CubeCoordinate, (u8, u8)>,
}

/// Multipler for output production rate of quarrys. Tweak this for game balancing
const PRODUCTION_RATE: f32 = 5.0;

/// Main function that calculates which quarry uses which stoney tile.
#[allow(clippy::type_complexity, clippy::too_many_arguments)]
pub fn calc_quarry_tile(
    map: Res<Map>,
    #[cfg(feature = "graphics")] mut debug_text: Query<
        &mut DebugText,
        Or<(
            With<Enum!(TileType::StoneQuarry)>,
            With<Enum!(TileType::StoneRocks)>,
            With<Enum!(TileType::StoneHill)>,
            With<Enum!(TileType::StoneMountain)>,
        )>,
    >,
    mut stone_tiles: Query<
        &mut RelevantTileMarker,
        (
            Or<(
                With<Enum!(TileType::StoneRocks)>,
                With<Enum!(TileType::StoneHill)>,
                With<Enum!(TileType::StoneMountain)>,
            )>,
            Without<Enum!(TileType::StoneQuarry)>,
        ),
    >,
    mut quarrys: Query<
        (&CubeCoordinate, &mut MetricsRate, &mut RelevantTileMarker),
        With<Enum!(TileType::StoneQuarry)>,
    >,
    rocks: Query<&CubeCoordinate, With<Enum!(TileType::StoneRocks)>>,
    hills: Query<&CubeCoordinate, With<Enum!(TileType::StoneHill)>>,
    mountains: Query<&CubeCoordinate, With<Enum!(TileType::StoneMountain)>>,
) {
    // hashmap size depending on tiles placed in map
    let hashmap_size = map.len();

    // final collection for which quarry uses which stoney tile
    let mut output: Vec<QuarryStoney> = Vec::new();
    // stoney-tile-cords, (free, max)
    let mut stoney_output: HashMap<CubeCoordinate, (u8, u8)> = HashMap::with_capacity(hashmap_size);

    // initialize stoney hashmaps
    let mut rocks_list: HashMap<CubeCoordinate, u8> = HashMap::with_capacity(hashmap_size);
    for &coords in rocks.iter() {
        rocks_list.insert(coords, 1);
        stoney_output.insert(coords, (1, 1));
    }

    let mut hills_list: HashMap<CubeCoordinate, u8> = HashMap::with_capacity(hashmap_size);
    for &coords in hills.iter() {
        hills_list.insert(coords, 2);
        stoney_output.insert(coords, (2, 2));
    }

    let mut mountains_list: HashMap<CubeCoordinate, u8> = HashMap::with_capacity(hashmap_size);
    for &coords in mountains.iter() {
        mountains_list.insert(coords, 3);
        stoney_output.insert(coords, (3, 3));
    }

    // initialize quarry hashset
    let mut quarry_list: HashSet<CubeCoordinate> = HashSet::with_capacity(hashmap_size);
    for (&coords, _, _) in quarrys.iter() {
        quarry_list.insert(coords);
    }

    remove_lonely_stoneys(
        &mut mountains_list,
        &mut hills_list,
        &mut rocks_list,
        &quarry_list,
        true,
    );

    trivial_solve(
        &mut rocks_list,
        &mut hills_list,
        &mut mountains_list,
        &mut quarry_list,
        &mut output,
        &mut stoney_output,
    );

    let best_solution: Solution = if quarry_list.is_empty() {
        // the problem was trivial
        debug!("Trivial Solve");
        Solution {
            production: evaluate_solution(&output),
            output,
            stoney_output,
        }
    } else {
        // All trivial cases done and no solution found
        // now brute force try all possibilities and take the (first) best solution
        debug!("Now this is complicated");

        let mountains_vec = from_hashmap_to_vec(&mountains_list);
        let hills_vec = from_hashmap_to_vec(&hills_list);
        let rocks_vec = from_hashmap_to_vec(&rocks_list);
        let quarry_vec = from_hashset_to_vec(&quarry_list);

        let max_best_production =
            calculate_max_best_production(quarry_vec.len(), &mountains_vec, &hills_vec, &rocks_vec);
        let cur_best_solution: Solution = Solution {
            production: 0,
            output: Vec::<QuarryStoney>::new(),
            stoney_output: HashMap::<CubeCoordinate, (u8, u8)>::new(),
        };

        complex_solve(
            quarry_vec,
            mountains_vec,
            hills_vec,
            rocks_vec,
            cur_best_solution,
            &output,
            &stoney_output,
            max_best_production,
        )
    };

    for mut relevant in stone_tiles.iter_mut() {
        relevant.reset();
    }

    // display quarry-production
    for quarry in best_solution.output.iter() {
        let production_rate = quarry.r as f32 * PRODUCTION_RATE;
        if let Some(Ok((_, mut rate, mut relevant))) = map.get(quarry.q).map(|e| quarrys.get_mut(e))
        {
            relevant.reset();
            if let Some(coord) = quarry.s {
                relevant.add(coord, RelKind::Primary);
                if let Some(Ok(mut relevant_stones)) =
                    map.get(coord).map(|e| stone_tiles.get_mut(e))
                {
                    relevant_stones.add(quarry.q, RelKind::Primary);
                }
            }
            *rate = MetricsRate {
                food: 0.0,
                materials: production_rate,
                money: 0.0,
            };
        }
        #[cfg(feature = "graphics")]
        if let Some(Ok(mut t)) = map.get(quarry.q).map(|e| debug_text.get_mut(e)) {
            t.add_section(
                "Quarry Tile:",
                format!("Rocklevel: {} \nproduces: {:.2}", quarry.r, production_rate),
            );
        }
    }

    // display stoney-slots
    #[cfg(feature = "graphics")]
    for (coord, (free, max)) in best_solution.stoney_output.iter() {
        if let Some(Ok(mut t)) = map.get(*coord).map(|e| debug_text.get_mut(e)) {
            t.add_section(
                "Stoney Tile:",
                format!("Free/max slots: \n({}/{})", *free, *max),
            );
        }
    }
}

/// removes all stoney-tiles from their corresponding HashMap if they have no unassigned quarrys adjacent.
///
/// quarry_list must contain all and only unassigned quarrys.
fn remove_lonely_stoneys(
    mountains_list: &mut HashMap<CubeCoordinate, u8>,
    hills_list: &mut HashMap<CubeCoordinate, u8>,
    rocks_list: &mut HashMap<CubeCoordinate, u8>,
    quarry_list: &HashSet<CubeCoordinate>,
    do_mountains_too: bool,
) {
    remove_lonely_stoneys_helper(hills_list, quarry_list);
    remove_lonely_stoneys_helper(rocks_list, quarry_list);

    // doesn't do anything if mountains_temp_reg in trivial_solve was done before
    if do_mountains_too {
        remove_lonely_stoneys_helper(mountains_list, quarry_list);
    }
}

/// helper for remove_lonely_stoneys
fn remove_lonely_stoneys_helper(
    stoneys_list: &mut HashMap<CubeCoordinate, u8>,
    quarry_list: &HashSet<CubeCoordinate>,
) {
    let mut to_be_removed: Vec<CubeCoordinate> = Vec::new();
    for (&stoney, _j) in stoneys_list.iter() {
        if stoney
            .get_neighbors()
            .filter(|x| quarry_list.contains(x))
            .count()
            == 0
        {
            to_be_removed.push(stoney);
        }
    }

    for lonely_stoney in to_be_removed.iter() {
        stoneys_list.remove(lonely_stoney);
    }
}

/// Solves trivial assignments of quarrys to stoney tiles
///
/// Quarrys in this context are not yet assigned quarrys in quarry_list.
/// The following trivial cases are being solved:
/// #### 1. Quarry has no neighboring stoney-tile
/// This quarry won't produce anything
/// #### 2. Quarry has exactly one adjacent stoney-tile
/// Quarry is assigned to this stoney-tile.
/// #### 3. Registration list
/// See also solve_via_registration_list.
fn trivial_solve(
    rocks_list: &mut HashMap<CubeCoordinate, u8>,
    hills_list: &mut HashMap<CubeCoordinate, u8>,
    mountains_list: &mut HashMap<CubeCoordinate, u8>,
    quarry_list: &mut HashSet<CubeCoordinate>,
    output: &mut Vec<QuarryStoney>,
    stoney_output: &mut HashMap<CubeCoordinate, (u8, u8)>,
) {
    // repeat until there are no more trivial solves found
    let mut trivial_solve_found = true;
    while trivial_solve_found {
        trivial_solve_found = false;

        let mut quarrys_to_be_deleted: Vec<CubeCoordinate> = Vec::new();

        for quarry in quarry_list.iter() {
            let stoney_neighbor_count = quarry
                .get_neighbors()
                .filter(|x| {
                    rocks_list.contains_key(x)
                        || hills_list.contains_key(x)
                        || mountains_list.contains_key(x)
                })
                .count();

            if stoney_neighbor_count == 0 {
                // 1. quarry doesn't produce anything -> mark as done
                output.push(QuarryStoney {
                    q: (*quarry),
                    s: (None),
                    r: 0,
                });
                quarrys_to_be_deleted.push(*quarry);
            } else if stoney_neighbor_count == 1 {
                // 2. quarry has only one adjacent stoney-tile -> assign to that
                for quarry_neighbor in quarry.get_neighbors() {
                    if let Some((free_slots, rocklvl)) =
                        is_stoney_neighbor(&quarry_neighbor, mountains_list, hills_list, rocks_list)
                    {
                        output.push(QuarryStoney {
                            q: (*quarry),
                            s: Some(quarry_neighbor),
                            r: rocklvl,
                        });
                        quarrys_to_be_deleted.push(*quarry);

                        if rocklvl == 1 {
                            rocks_list.remove(&quarry_neighbor);
                        } else if rocklvl == 2 {
                            if free_slots > 1 {
                                hills_list.insert(quarry_neighbor, free_slots - 1);
                            } else {
                                hills_list.remove(&quarry_neighbor);
                            }
                        } else if rocklvl == 3 {
                            if free_slots > 1 {
                                mountains_list.insert(quarry_neighbor, free_slots - 1);
                            } else {
                                mountains_list.remove(&quarry_neighbor);
                            }
                        }
                        stoney_output.insert(quarry_neighbor, (free_slots - 1, rocklvl));
                        break;
                    }
                }
            }
        }

        if !quarrys_to_be_deleted.is_empty() {
            trivial_solve_found = true;
        }
        for quarry in quarrys_to_be_deleted {
            quarry_list.remove(&quarry);
        }

        // all quarrys are assigned -> solved!
        if quarry_list.is_empty() {
            return;
        }

        // 3. solve via registration list
        // for mountains
        trivial_solve_found = solve_via_registration_list(
            mountains_list,
            &HashMap::new(),
            &HashMap::new(),
            quarry_list,
            output,
            stoney_output,
            3,
            trivial_solve_found,
        );
        // for hills
        trivial_solve_found = solve_via_registration_list(
            hills_list,
            mountains_list,
            &HashMap::new(),
            quarry_list,
            output,
            stoney_output,
            2,
            trivial_solve_found,
        );
        // for rocks
        trivial_solve_found = solve_via_registration_list(
            rocks_list,
            mountains_list,
            hills_list,
            quarry_list,
            output,
            stoney_output,
            1,
            trivial_solve_found,
        );
    }

    remove_lonely_stoneys(mountains_list, hills_list, rocks_list, quarry_list, false);
}

/// Quarrys attempt to register for adjacent stoney tiles in stoney_list.
///
/// If a stoney-tile has less or equal registrations than free slots there are no conflicts.
/// If there is no other, better option for a quarry they will get assigned to this stoney tile.
///
/// This is simple for mountain-tiles where you only have to look at the free-slots since there is no better option.
#[allow(clippy::too_many_arguments)]
fn solve_via_registration_list(
    stoney_list: &mut HashMap<CubeCoordinate, u8>,
    mountains_list: &HashMap<CubeCoordinate, u8>,
    hills_list: &HashMap<CubeCoordinate, u8>,
    quarry_list: &mut HashSet<CubeCoordinate>,
    output: &mut Vec<QuarryStoney>,
    stoney_output: &mut HashMap<CubeCoordinate, (u8, u8)>,
    rock_level: u8,
    mut trivial_solve_found: bool,
) -> bool {
    let mut stoney_temp_reg: HashMap<CubeCoordinate, WantMe> = HashMap::new();

    for i in stoney_list.iter() {
        stoney_temp_reg.insert(
            *i.0,
            WantMe {
                list: Vec::new(),
                num_free_slots: *i.1 as i8,
            },
        );
    }
    // register quarrys for all adjacent mountain tiles
    for quarry in quarry_list.iter() {
        // There are still free mountain slots adjacent

        for quarry_neighbor in quarry.get_neighbors() {
            if let Some(stoney_reg) = stoney_temp_reg.get_mut(&quarry_neighbor) {
                // potential quarry candidate for this mountain
                stoney_reg.list.push(*quarry);
                stoney_reg.num_free_slots -= 1;
            }
        }
    }

    debug!("Stoney registrations List, Level {}", rock_level);
    for i in stoney_temp_reg.iter() {
        debug!("{:?}", i);
    }

    for stoney_reg in stoney_temp_reg.iter() {
        // fewer neighboring quarrys than free slots (or no adjacent quarrys at all)
        if stoney_reg.1.num_free_slots >= 0 {
            let mut expired_reg = 0;
            let mut hope_for_better_stoney = 0;
            let mut something_actually_happened = false;

            // iterate registered quarrys
            for reg_quarry in stoney_reg.1.list.iter() {
                // this quarry has already found a partner
                if !quarry_list.contains(reg_quarry) {
                    expired_reg += 1;
                    continue;
                }

                // this quarry has adjacent stoney-tiles with higher rock-level
                if rock_level < 3
                    && reg_quarry
                        .get_neighbors()
                        .any(|x| mountains_list.contains_key(&x))
                {
                    hope_for_better_stoney += 1;
                    continue;
                }
                if rock_level < 2
                    && reg_quarry
                        .get_neighbors()
                        .any(|x| hills_list.contains_key(&x))
                {
                    hope_for_better_stoney += 1;
                    continue;
                }

                output.push(QuarryStoney {
                    q: (*reg_quarry),
                    s: Some(*stoney_reg.0),
                    r: rock_level,
                });
                // once quarry is assigned removed it from pool
                debug!(
                    "Solved something via registration List level {}",
                    rock_level
                );
                quarry_list.remove(reg_quarry);
                something_actually_happened = true;
            }

            // actually assigns quarry to stoney tile
            if something_actually_happened {
                stoney_list.remove(stoney_reg.0);
                stoney_output.insert(
                    *stoney_reg.0,
                    (
                        stoney_reg.1.num_free_slots as u8 + expired_reg + hope_for_better_stoney,
                        rock_level,
                    ),
                );
                trivial_solve_found = true;
            }
        }
    }
    trivial_solve_found
}

/// Returns if neighbor is a stoney tile and which type it is.
///
/// returns Option<(free_slots, rock_level)>
fn is_stoney_neighbor(
    neighbor: &CubeCoordinate,
    mountains_list: &HashMap<CubeCoordinate, u8>,
    hills_list: &HashMap<CubeCoordinate, u8>,
    rocks_list: &HashMap<CubeCoordinate, u8>,
) -> Option<(u8, u8)> {
    if let Some(_rocks) = rocks_list.get(neighbor) {
        return Some((1, 1));
    }
    if let Some(hills) = hills_list.get(neighbor) {
        return Some((*hills, 2));
    }
    if let Some(mountains) = mountains_list.get(neighbor) {
        return Some((*mountains, 3));
    }
    None
}

/// Calculates production rate of given solution.
///
/// Also works for none complete outputs.
fn evaluate_solution(output: &[QuarryStoney]) -> usize {
    let mut production: usize = 0;
    for i in output.iter() {
        production += i.r as usize;
    }
    production
}

/// Calculates maximum production based on total number of quarrys on entire map.
///
/// Begins by assigning slots to mountains, then hills and finally rocks.
fn calculate_max_best_production(
    mut num_quarrys: usize,
    mountains: &[(CubeCoordinate, u8)],
    hills: &[(CubeCoordinate, u8)],
    rocks: &[(CubeCoordinate, u8)],
) -> usize {
    let mut out = 0;
    out += calculate_max_best_production_helper(&mut num_quarrys, mountains, 3);
    out += calculate_max_best_production_helper(&mut num_quarrys, hills, 2);
    out += calculate_max_best_production_helper(&mut num_quarrys, rocks, 1);
    out
}

/// Helper function for calculate_max_best_production.
fn calculate_max_best_production_helper(
    num_quarrys: &mut usize,
    stoney: &[(CubeCoordinate, u8)],
    rock_level: usize,
) -> usize {
    let mut out = 0;
    if *num_quarrys == 0 {
        return out;
    }
    for &(_i, j) in stoney.iter() {
        let a = j as usize;
        if *num_quarrys >= a {
            out += a * rock_level;
            *num_quarrys -= a;
        } else if *num_quarrys == 0 {
            return out;
        } else {
            out += *num_quarrys * rock_level;
            return out;
        }
    }
    out
}

/// Function tries to solve current situation by guessing one quarry assignment (trying out all possible) to a stoney tile.
///
/// If this didn't solve the problem (still not trivial): recursive descent via all_quarrys_left -> guess next quarry
#[allow(clippy::too_many_arguments)]
fn complex_solve(
    mut all_quarrys_left: Vec<CubeCoordinate>,
    mountains: Vec<(CubeCoordinate, u8)>,
    hills: Vec<(CubeCoordinate, u8)>,
    rocks: Vec<(CubeCoordinate, u8)>,
    mut cur_best_solution: Solution,
    output: &[QuarryStoney],
    stoney_output: &HashMap<CubeCoordinate, (u8, u8)>,
    max_best_production: usize,
) -> Solution {
    // a best solution was found
    if cur_best_solution.production == max_best_production {
        error!("Best solution found to late");
        return cur_best_solution;
    }

    // break if local(in this recursive arm) max_best_production <= best_solution.production
    if evaluate_solution(output)
        + calculate_max_best_production(all_quarrys_left.len(), &mountains, &hills, &rocks)
        <= cur_best_solution.production
    {
        debug!("This is worse than what I've already seen and it won't get better. Let's get out of here");
        return cur_best_solution;
    }

    // this should not happen because it is a trivial solve
    if all_quarrys_left.is_empty() {
        error!("trivial solve was found too late");
    }

    // this should not happen because it is a trivial solve
    if all_quarrys_left.len() == 1 {
        error!("trivial solve was found too late");
    }

    let mut mountains_list = from_stoney_vec_to_hashmap(&mountains);
    let mut hills_list = from_stoney_vec_to_hashmap(&hills);
    let mut rocks_list = from_stoney_vec_to_hashmap(&rocks);

    // take one quarry and try all of its possibilities
    let quarry = all_quarrys_left.pop().unwrap();

    debug!("Let me guess something for {:?}", quarry);

    for quarry_neighbor in quarry.get_neighbors() {
        let mut something_happened: bool = false;
        let mut output_copy = output.to_owned();
        let mut stoney_output_copy = stoney_output.clone();

        // assigns current guess
        if let Some((free_slots, rocklvl)) =
            is_stoney_neighbor(&quarry_neighbor, &mountains_list, &hills_list, &rocks_list)
        {
            output_copy.push(QuarryStoney {
                q: quarry,
                s: Some(quarry_neighbor),
                r: rocklvl,
            });
            if rocklvl == 1 {
                rocks_list.remove(&quarry_neighbor);
            } else if rocklvl == 2 {
                if free_slots > 1 {
                    hills_list.insert(quarry_neighbor, free_slots - 1);
                } else {
                    hills_list.remove(&quarry_neighbor);
                }
            } else if rocklvl == 3 {
                if free_slots > 1 {
                    mountains_list.insert(quarry_neighbor, free_slots - 1);
                } else {
                    mountains_list.remove(&quarry_neighbor);
                }
            }
            stoney_output_copy.insert(quarry_neighbor, (free_slots - 1, rocklvl));
            something_happened = true;
        }

        if something_happened {
            complex_solve_helper(
                &all_quarrys_left,
                &mut rocks_list,
                &mut hills_list,
                &mut mountains_list,
                &mut output_copy,
                &mut stoney_output_copy,
                &mut cur_best_solution,
                max_best_production,
            );
            // a best solution was found
            if cur_best_solution.production == max_best_production {
                debug!("Best Solution found. I'm done here!");
                return cur_best_solution;
            }
        }
    }

    // try mapping this quarry to no stoney tile (no production)
    let mut output_copy = output.to_owned();
    let mut stoney_output_copy = stoney_output.clone();
    output_copy.push(QuarryStoney {
        q: quarry,
        s: None,
        r: 0,
    });

    complex_solve_helper(
        &all_quarrys_left,
        &mut rocks_list,
        &mut hills_list,
        &mut mountains_list,
        &mut output_copy,
        &mut stoney_output_copy,
        &mut cur_best_solution,
        max_best_production,
    );
    cur_best_solution
}

/// Helper for complex_solve.
/// Tries to simplify by executing trivial_solve.
/// If no solution is found yet it will execute complex_solve again (recursion).
///
/// Sets cur_best_solution if the found solution is better.
#[allow(clippy::too_many_arguments)]
fn complex_solve_helper(
    all_quarrys_left: &[CubeCoordinate],
    rocks_list: &mut HashMap<CubeCoordinate, u8>,
    hills_list: &mut HashMap<CubeCoordinate, u8>,
    mountains_list: &mut HashMap<CubeCoordinate, u8>,
    output_copy: &mut Vec<QuarryStoney>,
    stoney_output_copy: &mut HashMap<CubeCoordinate, (u8, u8)>,
    cur_best_solution: &mut Solution,
    max_best_production: usize,
) {
    let mut quarry_list = from_quarry_vec_to_hashset(all_quarrys_left);
    // simplify problem with trivial solve
    trivial_solve(
        rocks_list,
        hills_list,
        mountains_list,
        &mut quarry_list,
        output_copy,
        stoney_output_copy,
    );

    let tmp_solution = if quarry_list.is_empty() {
        // the remaining problem was trivial
        Solution {
            production: evaluate_solution(output_copy),
            output: output_copy.clone(),
            stoney_output: stoney_output_copy.clone(),
        }
    } else {
        // still a complex problem
        let solution = complex_solve(
            from_hashset_to_vec(&quarry_list),
            from_hashmap_to_vec(mountains_list),
            from_hashmap_to_vec(hills_list),
            from_hashmap_to_vec(rocks_list),
            cur_best_solution.clone(),
            output_copy,
            stoney_output_copy,
            max_best_production,
        );

        Solution {
            production: evaluate_solution(output_copy),
            output: solution.output,
            stoney_output: solution.stoney_output,
        }
    };
    if tmp_solution.production > cur_best_solution.production {
        cur_best_solution.production = tmp_solution.production;
        cur_best_solution.output = tmp_solution.output;
        cur_best_solution.stoney_output = tmp_solution.stoney_output;
    }
}

/// Helper function that puts all elements of hashmap into a vec.
///
/// Allows to dynamically shrink size of the hashmaps when the problem gets smaller.
fn from_hashmap_to_vec(stoney: &HashMap<CubeCoordinate, u8>) -> Vec<(CubeCoordinate, u8)> {
    let mut out: Vec<(CubeCoordinate, u8)> = Vec::new();
    for (&i, &j) in stoney.iter() {
        out.push((i, j));
    }
    out
}

/// Helper function that puts all elements of hashset into a vec.
///
/// Allows to dynamically shrink size of the hashsets when the problem gets smaller.
fn from_hashset_to_vec(quarry: &HashSet<CubeCoordinate>) -> Vec<CubeCoordinate> {
    let mut out: Vec<CubeCoordinate> = Vec::new();
    for &i in quarry.iter() {
        out.push(i);
    }
    out
}

/// Helper function that puts all elements of Vec into a HashMap.
///
/// Allows to dynamically shrink size of the hashmaps when the problem gets smaller.
fn from_stoney_vec_to_hashmap(stoney: &[(CubeCoordinate, u8)]) -> HashMap<CubeCoordinate, u8> {
    let mut out: HashMap<CubeCoordinate, u8> = HashMap::with_capacity(stoney.len() * 2);
    for (i, j) in stoney.iter() {
        out.insert(*i, *j);
    }
    out
}

/// Helper function that puts all elements of Vec into a Hashset.
///
/// Allows to dynamically shrink size of the hashsets when the problem gets smaller.
fn from_quarry_vec_to_hashset(quarry: &[CubeCoordinate]) -> HashSet<CubeCoordinate> {
    let mut out: HashSet<CubeCoordinate> = HashSet::with_capacity(quarry.len() * 2);
    for i in quarry.iter() {
        out.insert(*i);
    }
    out
}
