use bevy::prelude::*;
use bevy_enum_filter::Enum;
use serde::{Deserialize, Serialize};

use crate::{
    coordinates::CubeCoordinate,
    game::{
        map::Map,
        metrics::MetricsRate,
        report::Warnings,
        resources::RelKind,
        tiles::{tile_type_filters, TileType},
    },
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

use super::{
    collect_resource_rates, ConfigureMarketReq, MarketConfiguredEvent, MarketInfo,
    RelevantTileMarker,
};

// ------------- Change this for Game Balancing -------------
/// Radius in which the market can collect resources from other tiles.
/// This Number must be smaller than MIN_RADIUS in placer.rs
const COLLECT_RADIUS: usize = 3;
/// Radius in which houses boost this marketplace.
const BOOST_RADIUS: usize = 1;
/// Trade efficiency of a Marketplace without any houses in BOOST_RADIUS
const BASE_TRADE_EFFICIENCY: f32 = 0.2;
/// How much a houses improve the trade efficiency of a marketplace
const IMPROVE_BASE_RATIO_PER_CAPITA: f32 = 0.4;
// ------------- Change this for Game Balancing -------------

/// How much of the collected Resources will be converted to money
///
/// 0.0 -> Nothing gets sold
///
/// 1.0 -> Everything gets sold
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct SellKeepRatio {
    pub food: f32,
    pub materials: f32,
}
impl Default for SellKeepRatio {
    fn default() -> Self {
        Self {
            food: 0.0,
            materials: 0.0,
        }
    }
}
impl SellKeepRatio {
    pub fn clamp(mut self) -> Self {
        self.food = self.food.clamp(0.0, 1.0);
        self.materials = self.materials.clamp(0.0, 1.0);
        self
    }
}

/// Amount of Resources that a given Marketplace collected in COLLECT_RADIUS
/// This amount will be converted to money based on SellRatio
#[derive(Clone, Default, Debug)]
pub struct ResourceAmount {
    pub food: f32,
    pub materials: f32,
}

#[allow(clippy::too_many_arguments)]
pub fn calc_marketplace_tile(
    map: Res<Map>,
    #[cfg(feature = "graphics")] mut debug_text: Query<
        &mut DebugText,
        With<Enum!(TileType::Marketplace)>,
    >,
    mut markets: Query<
        (
            Entity,
            &CubeCoordinate,
            &mut MarketInfo,
            &mut MetricsRate,
            &mut RelevantTileMarker,
        ),
        With<Enum!(TileType::Marketplace)>,
    >,
    smallhouses: Query<&CubeCoordinate, With<Enum!(TileType::SmallHouse)>>,
    doublehouses: Query<&CubeCoordinate, With<Enum!(TileType::DoubleHouse)>>,
    tile_rates: Query<&MetricsRate, (With<TileType>, Without<Enum!(TileType::Marketplace)>)>,
) {
    // calculate all trade_efficiency
    for (_entity, coord, mut market, mut rate, mut relevant) in markets.iter_mut() {
        relevant.reset();

        let boost_amount = map
            .get_area_with_coords(*coord, BOOST_RADIUS)
            .map(|(x, coords)| {
                if smallhouses.contains(x) {
                    relevant.add(coords, RelKind::Secondary);
                    1
                } else if doublehouses.contains(x) {
                    relevant.add(coords, RelKind::Secondary);
                    2
                } else {
                    0
                }
            })
            .sum::<i32>();

        market.trade_efficiency = 1.0_f32
            .min(boost_amount as f32 * IMPROVE_BASE_RATIO_PER_CAPITA + BASE_TRADE_EFFICIENCY);

        // find all neighbors and "collect" their resources
        let mut collector: ResourceAmount = ResourceAmount::default();

        for (tiles_around_a_marketplace, coord) in map.get_area_with_coords(*coord, COLLECT_RADIUS)
        {
            if let Ok(rate) = tile_rates.get(tiles_around_a_marketplace) {
                if rate.food + rate.materials > f32::EPSILON {
                    relevant.add(coord, RelKind::Primary);
                }
                collector.food += rate.food;
                collector.materials += rate.materials;
            }
        }
        market.available = collector;

        // sell
        let sold_food = market.available.food * market.trade.food;
        let sold_materials = market.available.materials * market.trade.materials;
        let generated_money = (sold_food + sold_materials) * market.trade_efficiency;

        *rate = MetricsRate {
            food: -sold_food,
            materials: -sold_materials,
            money: generated_money,
        };
        debug!("Market metrics: {:?}", *rate);

        #[cfg(feature = "graphics")]
        if let Ok(mut t) = debug_text.get_mut(_entity) {
            t.add_section(
                "Marketplace Tile:",
                format!(
                    "Collect amount:\n{:.2}F    {:.2}M\nSell amount:\n{:.2}F    {:.2}M\nTrade eff: {:.2} \nTrade ratio:\n{:.2}F    {:.2}M\nGen Money: {:.2}\n",
                    market.available.food,
                    market.available.materials,
                    sold_food,
                    sold_materials,
                    market.trade_efficiency,
                    market.trade.food,
                    market.trade.materials,
                    generated_money,
                ),
            );
        }
    }
}

/// Changes Metrics based on changes in Marketplace-Settings without doing the whole metrics calculation on tile-placed
///
/// processes ChangedTradeRatio-Events
pub fn apply_changes_from_trade_ratio(
    map: Res<Map>,
    mut ev_config: EventReader<ConfigureMarketReq>,
    mut succ_ev_config: EventWriter<MarketConfiguredEvent>,
    mut markets: Query<(&mut MarketInfo, &mut MetricsRate), With<Enum!(TileType::Marketplace)>>,
    mut warnings: EventWriter<Warnings>,
) {
    for ConfigureMarketReq { coord, ratio } in ev_config.read() {
        if let Some(Ok((mut market, mut rate))) = map.get(*coord).map(|e| markets.get_mut(e)) {
            // setup new ratio
            market.trade = ratio.clone().clamp();

            // recalculate rate
            let sold_food = market.available.food * market.trade.food;
            let sold_materials = market.available.materials * market.trade.materials;
            let generated_money = (sold_food + sold_materials) * market.trade_efficiency;

            *rate = MetricsRate {
                food: -sold_food,
                materials: -sold_materials,
                money: generated_money,
            };
            succ_ev_config.send(MarketConfiguredEvent {
                coord: *coord,
                ratio: ratio.clone(),
            });
            debug!("Market metrics: {:?}", *rate);
        } else {
            warnings.send(Warnings::market());
            warn!("There is no market at {:?}", coord);
        }
    }
}

pub fn re_collect_resource_rates(
    metrics: ResMut<MetricsRate>,
    metrics_rate: Query<&MetricsRate, With<TileType>>,
    mut succ_ev_config: EventReader<MarketConfiguredEvent>,
) {
    // Updates MetricsRates, only possible here due to scheduling in mod.rs
    if !succ_ev_config.is_empty() {
        succ_ev_config.clear();
        collect_resource_rates(metrics, metrics_rate);
    }
}

pub fn track_market_economy(mut markets: Query<(&MetricsRate, &mut MarketInfo)>) {
    for (rate, mut market) in markets.iter_mut() {
        market.economy += rate;
    }
}
