use bevy::prelude::*;
use bevy_enum_filter::Enum;

use crate::{
    coordinates::CubeCoordinate,
    game::{metrics::MetricsRate, resources::moai::BOOSTRANGE_MOAI, tiles::tile_type_filters},
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

use super::{BoosterTile, RelKind, RelevantTileMarker};

/// Base Production Rate of small_house. Tweak this for game balancing
const BASE_PRODUCTION_RATE: f32 = 2.0;

pub fn calc_smallhouse_tile(
    #[cfg(feature = "graphics")] mut debug_text: Query<
        &mut DebugText,
        With<Enum!(TileType::SmallHouse)>,
    >,
    mut houses: Query<
        (
            Entity,
            &CubeCoordinate,
            &mut MetricsRate,
            &mut RelevantTileMarker,
        ),
        With<Enum!(TileType::SmallHouse)>,
    >,
    moai: Query<(&CubeCoordinate, &BoosterTile), With<BoosterTile>>,
) {
    // iterate all SmallHouse tiles
    for (_entity, coord, mut rate, mut relevant) in houses.iter_mut() {
        relevant.reset();
        // find moai tile with highest boost value in range
        let max_booster = moai
            .iter()
            .filter(|(boost_coord, _)| coord.distance(**boost_coord) <= BOOSTRANGE_MOAI)
            .fold((1.0, None), |(max, coord), (new_coord, boost)| {
                if max > **boost {
                    (max, coord)
                } else {
                    (**boost, Some(new_coord))
                }
            });

        if let Some(coord) = max_booster.1 {
            relevant.add(*coord, RelKind::Primary);
        }

        let production_rate = BASE_PRODUCTION_RATE * max_booster.0;
        *rate = MetricsRate {
            food: 0.0,
            materials: 0.0,
            money: production_rate,
        };

        // display production rate

        #[cfg(feature = "graphics")]
        if let Ok(mut t) = debug_text.get_mut(_entity) {
            t.add_section(
                "SmallHouse Tile:",
                format!("produces: {:.1}", production_rate),
            );
        }
    }
}
