use bevy::prelude::*;
use bevy_enum_filter::Enum;

use crate::{
    coordinates::CubeCoordinate,
    game::{map::Map, metrics::MetricsRate, tiles::tile_type_filters},
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

use super::{RelKind, RelevantTileMarker};

/// Base Production of beehive tiles. Tweak this for game balancing
const BASE_PRODUCTION_RATE: f32 = 0.;
/// One over the natural logorithm for the base in the production rate formula. Smaller Base -> higher Production. Tweak this for game balancing
const LOGARITHM_FACTOR: f32 = 0.6;
/// Radius in which forest has positive effect on beehives. Tweak this for game balancing
const FOREST_RADIUS: usize = 2;
/// Radius in which wheat has positive effect on beehives. Tweak this for game balancing
const WHEAT_RADIUS: usize = 2;

pub fn calc_beehive_tile(
    map: Res<Map>,
    #[cfg(feature = "graphics")] mut debug_text: Query<
        &mut DebugText,
        With<Enum!(TileType::Beehive)>,
    >,
    // add other plants here later
    wheat: Query<&CubeCoordinate, With<Enum!(TileType::Wheat)>>,
    forest: Query<&CubeCoordinate, With<Enum!(TileType::Forest)>>,
    mut hives: Query<
        (
            Entity,
            &CubeCoordinate,
            &mut MetricsRate,
            &mut RelevantTileMarker,
        ),
        With<Enum!(TileType::Beehive)>,
    >,
) {
    // iterate all beehive tiles
    for (_entity, coord, mut rate, mut relevant) in hives.iter_mut() {
        relevant.reset();
        let mut forest_neighbors = 0;
        for (entity, coord) in map.get_area_with_coords(*coord, FOREST_RADIUS) {
            if forest.contains(entity) {
                relevant.add(coord, RelKind::Primary);
                forest_neighbors += 1;
            }
        }
        let mut wheat_neighbors = 0;
        for (entity, coord) in map.get_area_with_coords(*coord, WHEAT_RADIUS) {
            if wheat.contains(entity) {
                relevant.add(coord, RelKind::Primary);
                wheat_neighbors += 1;
            }
        }

        let production_rate = ((forest_neighbors as f32 + 1.0) * (wheat_neighbors as f32 + 1.0))
            .ln()
            * LOGARITHM_FACTOR
            + BASE_PRODUCTION_RATE;
        *rate = MetricsRate {
            food: production_rate,
            materials: 0.0,
            money: 0.0,
        };
        #[cfg(feature = "graphics")]
        if let Ok(mut t) = debug_text.get_mut(_entity) {
            t.add_section("Beehive Tile:", format!("produces: {:.1}", production_rate));
        }
    }
}
