use bevy::{ecs::system::RunSystemOnce, prelude::*};

use crate::{
    game::{
        gamemodes::{
            challenge::ChallengeGameState, creative::CreativeGameState, zen::ZenGameState,
        },
        time::Game,
    },
    prelude::still_in_state,
    DisplayMode,
};

use super::{
    bot::ActionEvents,
    build_area::setup_build_area,
    controls::bot_placer::{TileActionEvent, TileActionType},
    gamemodes::{
        campaign::levels::init_campaign_mode, challenge::init_challenge_mode,
        creative::init_creative_mode, zen::init_zen_mode, GameMode,
    },
    hand::{generate_hand, reset_deck},
    metrics::{reset_metrics, reset_target_metrics, Metrics},
    recorder::{Record, RecordActionType, RecordedTick},
    reset_game_state,
    resources::{ResourceTickEvent, RESOURCE_TICK_INTERVAL},
    rewards::{reset_rewards_timer, RewardReqEvent},
    round::{continue_timer, reset_round_resources, update_redraw_cost, ManualRedrawEvent},
    time::{reset_game_time, unpause_game_time, GameTime, StepEvent},
    GameSimSet,
};

const NANOS_PER_MILLI: u32 = 1_000_000;

macro_rules! define_replayer_plugin {
    ($({
        // game mode to run in
        gamemode: $gm:expr,
        // parent gamemode of state
        state: $state:expr,
        // add run condition set is true when replayer should be ticked.
        tick_condition: $tc:expr,
    }),*) => {
        pub struct ReplayPlugin;
        impl Plugin for ReplayPlugin {
            fn build(&self, app: &mut App) {
                app.init_resource::<ReplayStep>();

                use crate::prelude::never;

                app.add_systems(
                    Update,
                    simulate_record
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(resource_exists::<Replayer>)
                        .in_set(GameSimSet::Receive),
                );

                app.add_systems(
                    Update,
                    schedule_simulate_record
                        .run_if(never()$(.or(still_in_state($state).and($tc)))*)
                        .run_if(resource_exists::<Replayer>)
                        .in_set(GameSimSet::NextAuto),
                );

                $(
                    app.add_systems(OnExit($gm), clean_replay);
                    app.add_systems(
                        OnEnter($state),
                        schedule_simulate_record
                            .after(continue_timer)
                            .after(update_redraw_cost)
                            .after(reset_round_resources)
                            .after(generate_hand)
                            .after(reset_deck)
                            .after(reset_game_state)
                            .after(reset_game_time)
                            .after(reset_metrics)
                            .after(unpause_game_time)
                            .after(setup_build_area)
                            .after(reset_target_metrics)
                            .after(reset_rewards_timer)
                            .run_if(resource_exists::<Replayer>),
                    );
                )*
            }
        }
    };
}

define_replayer_plugin![
    {
        gamemode: GameMode::Challenge,
        state: ChallengeGameState::InRound,
        tick_condition: on_event::<ResourceTickEvent>,
    },
    {
        gamemode: GameMode::Creative,
        state: CreativeGameState::InGame,
        tick_condition: |time: Res<Time<Game>>| !time.delta().is_zero(),
    },
    {
        gamemode: GameMode::Zen,
        state: ZenGameState::InRound,
        tick_condition: on_event::<ResourceTickEvent>,
    }
];

#[derive(Debug, Clone, Resource, Deref, DerefMut)]
pub struct Replay(pub Record);

#[derive(Debug, Resource)]
pub struct Replayer {
    pub results: Option<Metrics>,
    /// first action is last element
    pub data: Vec<RecordedTick>,
}

#[derive(Debug, Default, Resource, Deref, DerefMut)]
pub struct ReplayStep(pub Option<RecordedTick>);

/// starts replay, requires selected mode to be completely setup.
/// this system just fires one-shot-system for the selected mode.
pub fn start_replay(mut commands: Commands, replay: Res<Replay>) {
    let mode = replay.mode.clone();
    commands.queue(move |w: &mut World| {
        match mode {
            GameMode::Challenge => w.run_system_once(init_challenge_mode).unwrap(),
            GameMode::Creative => w.run_system_once(init_creative_mode).unwrap(),
            GameMode::Campaign => w.run_system_once(init_campaign_mode).unwrap(),
            GameMode::Zen => w.run_system_once(init_zen_mode).unwrap(),
        };
    });
    let mut data = replay.data.clone();
    data.reverse();

    commands.insert_resource(Replayer {
        results: replay.metrics.clone(),
        data,
    });
}

pub fn clean_replay(mut commands: Commands) {
    commands.remove_resource::<Replay>();
}

fn schedule_simulate_record(
    mut replay: ResMut<Replayer>,
    mut tick: ResMut<ReplayStep>,
    mut time_stepper: EventWriter<StepEvent>,
) {
    if let Some(tick) = tick.0.as_mut() {
        if tick.skipped > 0 {
            tick.skipped -= 1;

            // nothing to do: step time
            if tick.skipped > 0 {
                time_stepper.send(StepEvent(RESOURCE_TICK_INTERVAL));
            }
            return;
        }
    }
    if let Some(next_tick) = replay.data.pop() {
        *tick = ReplayStep(Some(next_tick));

        // reverse vector, because actions get popped
        tick.0
            .as_mut()
            .expect("Replay step should exists, was set to Some before.")
            .actions
            .reverse();

        if tick
            .0
            .as_ref()
            .expect("Replay step should exists, was set to Some before.")
            .skipped
            > 0
        {
            // nothing to do: step time
            time_stepper.send(StepEvent(RESOURCE_TICK_INTERVAL));
        }
    } else {
        time_stepper.send(StepEvent(RESOURCE_TICK_INTERVAL));
    }
}

fn simulate_record(
    mut tick: ResMut<ReplayStep>,
    time: Res<Time<Game>>,
    mut action_events: ActionEvents,
    mut time_stepper: EventWriter<StepEvent>,
    graphic_mode: Res<State<DisplayMode>>,
) {
    let mut reset = false;

    if let Some(tick) = tick.0.as_mut() {
        if tick.skipped == 0 {
            let todo = tick.actions.len();
            if todo > 0 {
                // current game time
                let sub_ms = time.sub_nanos() / NANOS_PER_MILLI;

                // next time an action is scheduled
                let next_ms = tick.actions.last().map(|f| f.1).unwrap_or(0);

                // amount not possible in left time, with worst case fps
                let overhead = todo as i64 - ((1000 - sub_ms as i64) / 100);

                // run if tick is half over
                // or action is schedule has arrived
                // or their is an overhead
                if sub_ms > 500
                    || next_ms <= sub_ms
                    || overhead > 0
                    || *graphic_mode.get() == DisplayMode::Headless
                {
                    for _ in 0..=overhead.max(0) {
                        // run actions
                        let Some(action) = tick.actions.pop() else {
                            continue;
                        };

                        match action.0 {
                            RecordActionType::Redraw => {
                                action_events.redraw.send(ManualRedrawEvent);
                            }
                            RecordActionType::PlaceTile(tile) => {
                                action_events
                                    .tile
                                    .send(TileActionEvent(TileActionType::Place(tile.clone())));
                            }
                            RecordActionType::TakeTile(coord) => {
                                action_events
                                    .tile
                                    .send(TileActionEvent(TileActionType::Take(coord)));
                            }
                            RecordActionType::CollectReward(coord) => {
                                action_events.reward.send(RewardReqEvent(coord));
                            }
                            RecordActionType::ConfigureMarketplaceTile(data) => {
                                action_events.market.send(data.clone().into());
                            }
                        };
                    }
                }
            }

            if tick.actions.is_empty() {
                reset = true;
            }
        }
    }

    if reset {
        time_stepper.send(StepEvent(RESOURCE_TICK_INTERVAL));
        **tick = None;
    }
}
