#[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
use std::{fs, io::Write, path};

use bevy::prelude::*;
use serde::{Deserialize, Serialize};

#[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
use crate::download;
use crate::{
    coordinates::CubeCoordinate,
    game::{
        gamemodes::{
            challenge::ChallengeGameState, creative::CreativeGameState, zen::ZenGameState,
        },
        GameSimSet,
    },
    GameConfig,
};

use super::{
    gamemodes::{
        campaign::{levels::CampaignGroupWrapper, CampaignModeConfig},
        challenge::ChallengeModeConfig,
        zen::ZenModeConfig,
        GameMode,
    },
    map::TileSaveData,
    metrics::Metrics,
    resources::{MarketConfiguredEvent, ResourceTickEvent, TilePlacedEvent, TileReclaimedEvent},
    rewards::RewardEvent,
    round::{Redraw, RoundCounter},
    time::{Game, GameTime},
};

const NANOS_PER_MILLI: u32 = 1_000_000;

macro_rules! define_recorder_plugin {
    ($({
        // in which sub game mode to run the record plugin
        gamemode: $gm:expr,
        // parent gamemode of state
        state: $state:expr,
        // add run condition set is true when recorder should be ticked.
        tick_condition: $tc:expr,
    }),*) => {
        pub struct RecorderPlugin;
        impl Plugin for RecorderPlugin {
            fn build(&self, app: &mut bevy::prelude::App) {
                app.init_resource::<Record>();
                app.add_event::<SaveRecord>();
                app.add_event::<SavedRecord>();

                use crate::prelude::*;

                $(app.add_systems(OnEnter($gm), start_recorder);)*

                app.add_systems(
                    Update,
                    step_recorder
                        .run_if(never()$(.or(in_state($gm).and($tc)))*)
                        .run_if(recorder_activated)
                        .after(GameSimSet::Simulation), // event is send before Simulation, so after Simulation is sufficient
                );

                app.add_systems(
                    Update,
                    receive_recorder
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(recorder_activated)
                        .after(GameSimSet::Prepare)
                        .before(step_recorder),
                        // has to run after step_recorder,
                        // because the next tick is started there,
                        // but we wanna receive data from the last tick.
                );

                app.add_systems(
                    Update,
                    save_record_evt.run_if(recorder_activated),
                );

                $(
                    app.add_systems(
                        OnExit($gm),
                        save_record_cli.run_if(recorder_activated),
                    );
                )*
                // HACK: save on exit.
                // this might not work if exit is too fast.
                // But should work, because it runs in PostUpdate, and is able to receive the exit event in the same tick as send.
                app.add_systems(
                    PostUpdate,
                    save_record_cli
                        .run_if(never()$(.or(in_state($gm)))*)
                        .run_if(on_event::<AppExit>)
                        .run_if(recorder_activated),
                );
            }
        }
    };
}

// adjust saved data in systems in this file
define_recorder_plugin![
    {
        gamemode: GameMode::Challenge,
        state: ChallengeGameState::InRound,
        tick_condition: on_event::<ResourceTickEvent>,
    },
    {
        gamemode: GameMode::Creative,
        state: CreativeGameState::InGame,
        tick_condition: |time: Res<Time<Game>>| !time.delta().is_zero(),
    },
    {
        gamemode: GameMode::Zen,
        state: ZenGameState::InRound,
        tick_condition: on_event::<ResourceTickEvent>,
    }
];

pub fn recorder_activated(config: Res<GameConfig>) -> bool {
    config.activate_recorder
}

#[derive(Debug, Event)]
pub struct SaveRecord(pub String);

#[derive(Debug, Event)]
/// Game saved, true is successful, false on error
/// Will only be send if `SaveRecord` was send.
pub struct SavedRecord(pub bool);

#[derive(Default, Clone, Debug, Resource, Serialize, Deserialize)]
pub struct Record {
    pub mode: GameMode,
    pub campaign: Option<CampaignGroupWrapper>,
    pub challenge: Option<ChallengeModeConfig>,
    pub zen: Option<ZenModeConfig>,
    #[serde(alias = "results")]
    pub metrics: Option<Metrics>,
    pub round: Option<RoundCounter>,
    pub data: Vec<RecordedTick>,
    #[serde(default)]
    pub name: Option<String>,
}

#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct RecordedTick {
    #[serde(rename = "s")]
    #[serde(alias = "skipped")]
    pub skipped: usize,

    #[serde(rename = "a")]
    #[serde(alias = "actions")]
    pub actions: Vec<RecordAction>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct RecordAction(pub RecordActionType, pub u32);

#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum RecordActionType {
    PlaceTile(TileSaveData),
    TakeTile(CubeCoordinate),
    CollectReward(CubeCoordinate),
    Redraw,
    ConfigureMarketplaceTile(MarketConfiguredEvent),
}

fn save_record(
    raw_path: &str,
    recorder: &mut ResMut<Record>,
    metrics: &Res<Metrics>,
    round: &Res<RoundCounter>,
) -> bool {
    if recorder.mode == GameMode::Challenge {
        recorder.metrics = Some((*metrics).clone());
        recorder.round = Some(RoundCounter(***round));
    } else {
        recorder.metrics = None;
    }
    match serde_json::to_string(&**recorder) {
        Ok(data) => {
            #[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
            {
                let path = path::Path::new(raw_path);
                if let Some(parent) = path.parent() {
                    // recursively create parent folders
                    match fs::create_dir_all(parent) {
                        Ok(_) => (),
                        Err(err) => {
                            warn!("Error while saving record: {}", err);
                            return false;
                        }
                    }
                }
                match fs::File::create(path).and_then(|mut file| file.write_all(data.as_bytes())) {
                    Ok(_) => {
                        info!("Saved game record to: {}", raw_path);
                        return true;
                    }
                    Err(err) => warn!("Error while saving record: {}", err),
                }
            }

            #[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
            {
                download(&data, raw_path, "text/json");
                return true;
            }
        }
        Err(err) => warn!("Error while saving the game record: {}", err),
    };
    false
}

fn save_record_evt(
    mut evt: EventReader<SaveRecord>,
    mut evt_res: EventWriter<SavedRecord>,
    mut recorder: ResMut<Record>,
    metrics: Res<Metrics>,
    round: Res<RoundCounter>,
) {
    for path in evt.read() {
        evt_res.send(SavedRecord(save_record(
            &path.0,
            &mut recorder,
            &metrics,
            &round,
        )));
    }
}

fn save_record_cli(
    mut recorder: ResMut<Record>,
    metrics: Res<Metrics>,
    config: Res<GameConfig>,
    round: Res<RoundCounter>,
) {
    if let Some(path) = &config.record_game {
        save_record(path, &mut recorder, &metrics, &round);
    }
    recorder.data.clear();
}

fn start_recorder(
    mut recorder: ResMut<Record>,
    game_mode: Res<State<GameMode>>,
    challenge_config: Option<Res<ChallengeModeConfig>>,
    zen_config: Option<Res<ZenModeConfig>>,
    campaign_config: Option<Res<CampaignModeConfig>>,
    config: Res<GameConfig>,
    mode: Res<State<GameMode>>,
) {
    recorder.mode = game_mode.get().clone();
    if **mode == GameMode::Challenge {
        // save challenge config
        recorder.challenge = challenge_config.map(|f| f.clone());
    } else {
        recorder.challenge = None;
    }
    if **mode == GameMode::Campaign {
        // save campaign config
        recorder.campaign = campaign_config.map(|f| f.level.clone());
    } else {
        recorder.campaign = None;
    }
    if **mode == GameMode::Zen {
        // save zen config
        recorder.zen = zen_config.map(|f| f.clone());
    } else {
        recorder.zen = None;
    }
    recorder.name = if config.replay_name.is_empty() {
        None
    } else {
        Some(config.replay_name.clone())
    };
    // reset recorder
    recorder.data.clear();
    // start first tick
    recorder.data.push(RecordedTick::default());
}

fn step_recorder(mut recorder: ResMut<Record>) {
    let Some(tick) = recorder.data.last_mut() else {
        debug!("Recorder in stall. Trying to write to recorder, without active tick.");
        return;
    };

    if tick.actions.is_empty() {
        tick.skipped += 1;
    } else {
        recorder.data.push(RecordedTick::default());
    }
}

fn receive_recorder(
    time: Res<Time<Game>>,
    mut recorder: ResMut<Record>,
    mut redraw: EventReader<Redraw>,
    mut place: EventReader<TilePlacedEvent>,
    mut market: EventReader<MarketConfiguredEvent>,
    mut take: EventReader<TileReclaimedEvent>,
    mut rewards: EventReader<RewardEvent>,
) {
    let Some(tick) = recorder.data.last_mut() else {
        debug!("Recorder in stall. Trying to write to recorder, without active tick.");
        return;
    };

    // record tile placements
    for tile in place.read() {
        tick.actions.push(RecordAction(
            RecordActionType::PlaceTile(tile.0.clone()),
            time.sub_nanos() / NANOS_PER_MILLI,
        ));
    }

    // record tile reclaim
    for tile in take.read() {
        tick.actions.push(RecordAction(
            RecordActionType::TakeTile(tile.0),
            time.sub_nanos() / NANOS_PER_MILLI,
        ));
    }

    // record markets
    if let Some(market) = market.read().last() {
        tick.actions
            .retain(|action| !matches!(action.0, RecordActionType::ConfigureMarketplaceTile(_)));
        tick.actions.push(RecordAction(
            RecordActionType::ConfigureMarketplaceTile(market.clone()),
            time.sub_nanos() / NANOS_PER_MILLI,
        ));
    }

    // record rewards
    for reward in rewards.read() {
        tick.actions.push(RecordAction(
            RecordActionType::CollectReward(reward.0),
            time.sub_nanos() / NANOS_PER_MILLI,
        ));
    }

    // record redraws
    for _ in redraw.read() {
        tick.actions.push(RecordAction(
            RecordActionType::Redraw,
            time.sub_nanos() / NANOS_PER_MILLI,
        ));
    }
}
