use std::time::Duration;

use bevy::prelude::*;

use crate::{
    coordinates::CubeCoordinate,
    game::gamemodes::{
        campaign::levels::{
            challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
        },
        challenge::ChallengeGameState,
        zen::ZenGameState,
    },
};

#[cfg(feature = "graphics")]
use super::visuals::placement_animation::set_animation_material;
use super::{
    hand::{Hand, HAND_SIZE},
    metrics::{Metrics, MetricsRate},
    tiles::{RemoveTile, TileType},
    time::Game,
    GameSimSet,
};

/// Metrics will be subtracted by rate * this
const RECLAIM_PENALTY_IN_SEC: f32 = 0.5;

macro_rules! define_reclaim_plugin {
    ($({
        // state in which reclaim should be generated
        state: $state:expr,
    }),*) => {
        pub struct PlacementReclaimPlugin;
        impl Plugin for PlacementReclaimPlugin {
            fn build(&self, app: &mut App) {
                use crate::prelude::never;

                #[cfg(feature = "graphics")]
                app.add_systems(
                    Update,
                    (
                        apply_placement_timer.after(set_animation_material),
                        tick_placement_timer,
                    )
                        .chain()
                        .in_set(GameSimSet::AfterSim)
                        .run_if(never()$(.or(in_state($state)))*),
                );

                #[cfg(not(feature = "graphics"))]
                app.add_systems(
                    Update,
                    (
                        apply_placement_timer,
                        tick_placement_timer,
                    )
                        .chain()
                        .in_set(GameSimSet::AfterSim)
                        .run_if(never()$(.or(in_state($state)))*),
                );
            }
        }
    };
}

define_reclaim_plugin![
    {
        state: ChallengeGameState::InRound,
    },
    {
        state: CampaignChallengeState::InRound,
    },
    {
        state: CampaignWettbewerbState::InRound,
    },
    {
        state: ZenGameState::InRound,
    }
];

#[derive(Component)]
pub struct PlacementTimer {
    pub timer: Timer,
}

#[derive(Component)]
pub struct TileLoaded;

impl Default for PlacementTimer {
    fn default() -> Self {
        PlacementTimer {
            timer: Timer::new(Duration::from_secs(5), bevy::time::TimerMode::Once),
        }
    }
}

#[allow(clippy::type_complexity)]
pub fn apply_placement_timer(
    mut commands: Commands,
    tiles: Query<
        Entity,
        (
            Added<TileType>,
            Without<PlacementTimer>,
            Without<TileLoaded>,
        ),
    >,
) {
    for e in &tiles {
        commands.entity(e).insert(PlacementTimer::default());
    }
}

pub fn tick_placement_timer(
    mut commands: Commands,
    mut tiles: Query<(Entity, &mut PlacementTimer)>,
    time: Res<Time<Game>>,
) {
    for (entity, mut animation) in &mut tiles {
        animation.timer.tick(time.delta());
        if !animation.timer.just_finished() {
            continue;
        }
        commands
            .entity(entity)
            .remove::<PlacementTimer>()
            .insert(TileLoaded);
    }
}

/// Handles tile reclaim.
/// Return true if tile was reclaimed.
pub fn handle_tile_reclaim(
    commands: &mut Commands,
    entity: Entity,
    loading_tiles: &Query<(&TileType, &CubeCoordinate), With<PlacementTimer>>,
    hand: &mut ResMut<Hand>,
    metrics: &mut ResMut<Metrics>,
    rate: &Res<MetricsRate>,
) -> bool {
    let Ok((tile, coord)) = loading_tiles.get(entity) else {
        return false;
    };

    if hand.items.len() >= HAND_SIZE {
        return false;
    }

    // reclaim tile
    hand.items.insert(0, *tile);
    commands.queue(RemoveTile::new(*coord));
    **metrics -= &((**rate).clone() * RECLAIM_PENALTY_IN_SEC);

    true
}
