pub mod asset_loading;
#[cfg(feature = "graphics")]
pub mod audio;
pub mod bot;
pub mod build_area;
pub mod controls;
pub mod gamemodes;
pub mod hand;
pub mod map;
pub mod metrics;
pub mod placement_reclaim;
pub mod resources;
mod rewards;
pub mod round;
pub mod tiles;

pub mod recorder;
pub mod replayer;
pub mod report;
pub mod time;
#[cfg(feature = "graphics")]
pub mod visuals;

use bevy::prelude::*;
use bot::BotPlugin;
use build_area::BuildAreaPlugin;
use gamemodes::GameModePlugin;
use hand::HandPlugin;
use map::{Map, MapPlugin};
use metrics::{MetricsPlugin, TargetMetricsPlugin};
use placement_reclaim::PlacementReclaimPlugin;
use recorder::RecorderPlugin;
use replayer::ReplayPlugin;
use report::ReportPlugin;
use resources::ResourceManagerPlugin;
use rewards::RewardsPlugin;
use round::RoundPlugin;
use serde::{Deserialize, Serialize};
use tiles::TilesPlugin;
use time::TimePlugin;

use crate::{errors::ErrorDisplay, AppState, DisplayMode, GameConfig};

/// Separate state to handle game play/pause modes
/// time based system should not run if the game is paused
/// ui based system may run, however they might interfere with the pause ui
///
/// structured as separate system to prevent interference with OnEnter
/// and keeps us from having to store the GameState, where we want to return to
#[derive(Debug, Clone, Copy, Default, PartialEq, Eq, Hash, SubStates)]
#[source(AppState = AppState::InGame)]
pub enum GamePauseState {
    /// the game is running
    #[default]
    Running,
    /// the game is paused
    Paused,
}

pub struct GamePlugin {
    display: DisplayMode,
}

impl GamePlugin {
    pub fn new(display: DisplayMode) -> Self {
        Self { display }
    }
}

impl Plugin for GamePlugin {
    fn build(&self, app: &mut App) {
        app.init_resource::<GameConfig>();
        app.init_resource::<Successful>();
        app.add_event::<GameEnd>();

        // display
        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display {
            app.add_plugins(visuals::VisualsPlugin);
            app.add_plugins(audio::AudioPlugin);
        }

        app.add_plugins(controls::ControlsPlugin {
            display_mode: self.display,
        });

        app.add_plugins(GameModePlugin {
            display_mode: self.display,
        });
        app.add_plugins(MapPlugin);

        app.add_plugins(BuildAreaPlugin);
        app.add_plugins(PlacementReclaimPlugin);

        app.add_plugins(TilesPlugin {
            #[cfg(feature = "graphics")]
            display_mode: self.display,
        });

        app.add_plugins(BotPlugin { mode: self.display });

        app.add_plugins(RecorderPlugin);
        app.add_plugins(ReplayPlugin);
        app.add_plugins(ReportPlugin);

        /////////

        app.add_systems(OnEnter(AppState::InGame), reset_game_state);
        app.add_sub_state::<GamePauseState>();

        if let DisplayMode::Headless = self.display {
            app.add_systems(OnEnter(AppState::Error), stop_on_error);

            app.add_systems(Update, headless_housekeeping);
        }

        app.insert_resource(Map::new());

        // register plugins
        app.add_plugins(TimePlugin {
            display: self.display,
        });
        app.add_plugins(ResourceManagerPlugin);
        app.add_plugins(RewardsPlugin {
            display_mode: self.display,
        });
        app.add_plugins(RoundPlugin {
            display_mode: self.display,
        });
        app.add_plugins(HandPlugin);

        app.add_plugins(MetricsPlugin);
        app.add_plugins(TargetMetricsPlugin);

        app.configure_sets(
            Update,
            (
                GameSimSet::Receive,
                GameSimSet::Prepare,
                GameSimSet::Simulation,
                GameSimSet::AfterSim,
                GameSimSet::NextAuto,
            )
                .chain(),
        );
    }
}

#[derive(Debug, Event)]
pub struct GameEnd;

#[derive(SystemSet, Debug, Clone, PartialEq, Eq, Hash)]
pub enum GameSimSet {
    /// Receive actions from bot or replay if present.
    Receive,

    /// Calculates:
    /// - placers (rng)
    /// - marketplace
    /// - rewards (rng)
    /// - redraw (rng)
    ///
    /// with internal conditions for deterministic dependencies (e.g. rng)
    /// all: sends events for excepted actions
    /// NOTE: receive_recorder
    Prepare,

    /// - Resource Pipe
    /// - Resource simulation
    ///
    /// runs code triggering `ResourceTickEvent`
    Simulation,

    /// calculates important thing for next tick
    /// - round_end_processor
    /// - update_redraw_cost
    /// - tick_placement_timer
    AfterSim,

    /// triggers next auto tools:
    /// - bot
    /// - replay
    ///
    /// content only runs, if still in state (internal conditions)
    NextAuto,
}

#[derive(Debug, Clone, Resource, Deref, DerefMut, Serialize, Deserialize)]
pub struct Successful(pub bool);

impl Default for Successful {
    fn default() -> Self {
        Self(true)
    }
}

fn headless_housekeeping(
    game_end: EventReader<GameEnd>,
    mut exit: EventWriter<AppExit>,
    succ: Res<Successful>,
) {
    if !game_end.is_empty() {
        info!("Game is over.");
        if **succ {
            exit.send(AppExit::Success);
        } else {
            exit.send(AppExit::error());
        }
    }
}

fn reset_game_state(mut map: ResMut<Map>) {
    *map = Map::new();
}

fn stop_on_error(errors: Query<&ErrorDisplay>, mut exit: EventWriter<AppExit>) {
    for ErrorDisplay {
        title,
        description,
        link,
    } in &errors
    {
        error!(
            "{title}: {description}{}",
            link.clone().map(|l| format!(" ({l})")).unwrap_or("".into())
        );
    }
    exit.send(AppExit::error());
}
