use bevy::prelude::*;
use serde::{Deserialize, Serialize};
use std::collections::HashMap;
use std::ops::{AddAssign, Mul, SubAssign};

use crate::coordinates::CubeCoordinate;

use super::gamemodes::campaign::levels::challenge::CampaignChallengeState;
use super::gamemodes::campaign::levels::wettbewerb::CampaignWettbewerbState;
use super::gamemodes::campaign::levels::CampaignGroups;
use super::gamemodes::challenge::ChallengeGameState;
use super::gamemodes::zen::ZenGameState;
use super::gamemodes::GameMode;
use super::round::RoundCounter;

macro_rules! define_metrics_plugin {
    ($({
       // in which game mode to run the reset system
       gamemode: $gm:expr$(,)?
    }),*) => {
        pub struct MetricsPlugin;
        impl Plugin for MetricsPlugin {
            fn build(&self, app: &mut App) {
                app.insert_resource(Metrics::default());
                app.insert_resource(MetricsRate::default());
                $(
                    app.add_systems(OnEnter($gm), reset_metrics);
                )*
            }
        }
    }
}
define_metrics_plugin![
    {
        gamemode: GameMode::Creative
    },
    {
        gamemode: GameMode::Challenge
    },
    {
        gamemode: CampaignGroups::CampaignChallenge
    },
    {
        gamemode: CampaignGroups::CampaignWettbewerb
    },
    {
        gamemode: GameMode::Zen
    }
];

/// The resources the user has collected.
#[derive(Resource, Default, Debug, Clone, PartialEq, Serialize, Deserialize)]
pub struct Metrics {
    pub food: f32,
    pub materials: f32,
    pub money: f32,
}
impl SubAssign<&Metrics> for Metrics {
    fn sub_assign(&mut self, rhs: &Metrics) {
        self.money -= rhs.money;
        self.materials -= rhs.materials;
        self.food -= rhs.food;
    }
}
impl PartialOrd<Metrics> for Metrics {
    fn partial_cmp(&self, other: &Metrics) -> Option<std::cmp::Ordering> {
        if self.money == other.money && self.materials == other.materials && self.food == other.food
        {
            return Some(std::cmp::Ordering::Equal);
        }
        if self.money >= other.money && self.materials >= other.materials && self.food >= other.food
        {
            return Some(std::cmp::Ordering::Greater);
        }
        Some(std::cmp::Ordering::Less)
    }
}

/// As resource: The current rate with which the user collects resources.
/// As Tile: The current rate with wich a tile produces resources.
#[derive(Resource, Component, Default, Debug, Clone, Serialize, Deserialize)]
pub struct MetricsRate {
    pub food: f32,
    pub materials: f32,
    pub money: f32,
}

impl AddAssign<&MetricsRate> for Metrics {
    fn add_assign(&mut self, rhs: &MetricsRate) {
        self.food += rhs.food;
        self.materials += rhs.materials;
        self.money += rhs.money;
    }
}
impl AddAssign<&MetricsRate> for MetricsRate {
    fn add_assign(&mut self, rhs: &MetricsRate) {
        self.food += rhs.food;
        self.materials += rhs.materials;
        self.money += rhs.money;
    }
}
impl SubAssign<&MetricsRate> for Metrics {
    fn sub_assign(&mut self, rhs: &MetricsRate) {
        self.food -= rhs.food;
        self.materials -= rhs.materials;
        self.money -= rhs.money;
    }
}
impl Mul<f32> for MetricsRate {
    type Output = MetricsRate;

    fn mul(self, rhs: f32) -> Self::Output {
        Self::Output {
            food: self.food * rhs,
            materials: self.materials * rhs,
            money: self.money * rhs,
        }
    }
}

pub fn reset_metrics(
    mut metrics: ResMut<Metrics>,
    mut metrics_rate: ResMut<MetricsRate>,
    mut rate_per_tile: Query<&mut MetricsRate>,
) {
    *metrics = Metrics::default();
    *metrics_rate = MetricsRate::default();
    for mut rate in rate_per_tile.iter_mut() {
        *rate = MetricsRate::default();
    }
}

#[derive(Resource, Default, Deref, DerefMut)]
pub struct ResourceRatePerTile(HashMap<CubeCoordinate, MetricsRate>);

macro_rules! define_target_metrics_plugin {
    ($({
    // in which game mode to run the reset system
        gamemode: $gm:expr,
        // in which substate of the game mode to increase the metrics on enter
        raise_state: $state:expr$(,)?
    }),*) => {
        pub struct TargetMetricsPlugin;
        impl Plugin for TargetMetricsPlugin {
            fn build(&self, app: &mut App) {
                app.insert_resource(TargetMetrics::default());
                $(
                    app.add_systems(OnEnter($state), raise_target_metrics);
                    app.add_systems(OnEnter($gm), reset_target_metrics);
                )*
            }
        }
    }
}
define_target_metrics_plugin![
    {
        gamemode: GameMode::Challenge,
        raise_state: ChallengeGameState::BetweenRounds
    },
    {
        gamemode: CampaignGroups::CampaignChallenge,
        raise_state: CampaignChallengeState::BetweenRounds
    },
    {
        gamemode: CampaignGroups::CampaignWettbewerb,
        raise_state: CampaignWettbewerbState::BetweenRounds
    },
    {
        gamemode: GameMode::Zen,
        raise_state: ZenGameState::BetweenRounds
    }
];

/// the resources the user has to collect
/// displayed as Metrics out of TargetMetrics
#[derive(Resource, Debug, Clone, Serialize, Deserialize)]
pub struct TargetMetrics {
    pub food: f32,
    pub materials: f32,
    pub money: f32,
}

const BASELINE_RATE: f32 = 60.0;

impl Default for TargetMetrics {
    fn default() -> Self {
        Self {
            food: BASELINE_RATE,
            materials: BASELINE_RATE,
            money: BASELINE_RATE,
        }
    }
}

fn raise_target_metrics(mut target: ResMut<TargetMetrics>, round: Res<RoundCounter>) {
    target.food = (**round as f32 + 1.0).powi(3) * BASELINE_RATE;
    target.materials = target.food;
    target.money = target.food;
}

pub fn reset_target_metrics(mut target_metrics: ResMut<TargetMetrics>) {
    *target_metrics = TargetMetrics::default();
}
