use std::collections::{HashMap, HashSet, LinkedList};

use bevy::{asset::AssetLoader, prelude::*};
use serde::{Deserialize, Serialize};
use thiserror::Error;

use crate::coordinates::CubeCoordinate;

use super::{
    resources::MarketInfo,
    tiles::{
        chameleon_tile::ChameleonTileData, rgb_tile::RgbTileData, AddTile, ConsistentTileData,
        TileType,
    },
};

/* define_asset_collection!(
    AssetWorlds,
    !test : GameWorld = "maps/test.json",
); */

pub struct MapPlugin;
impl Plugin for MapPlugin {
    fn build(&self, app: &mut App) {
        app.init_asset::<GameWorld>();
        //app.register_asset_collection::<AssetWorlds>();
        app.init_asset_loader::<SaveMapResourceLoader>();
    }
}

#[derive(Resource)]
pub struct Map {
    // uses Hexagon Cube Coordinates
    map: HashMap<CubeCoordinate, Entity>,
}

#[allow(dead_code)]
impl Default for Map {
    fn default() -> Self {
        Self::new()
    }
}

impl Map {
    pub fn new() -> Self {
        Self {
            map: HashMap::with_capacity(2048),
        }
    }

    pub fn get(&self, coords: CubeCoordinate) -> Option<Entity> {
        self.map.get(&coords).copied()
    }

    pub fn set(&mut self, coords: CubeCoordinate, entity: Entity) {
        self.map.insert(coords, entity);
    }

    pub fn remove(&mut self, coords: CubeCoordinate) {
        self.map.remove(&coords);
    }

    pub fn is_empty(&self) -> bool {
        self.map.is_empty()
    }

    pub fn len(&self) -> usize {
        self.map.len()
    }
    /// The iterator uses the Map to look up all entities so it can't out live the Map.
    pub fn get_all(&self) -> impl Iterator<Item = (&CubeCoordinate, &Entity)> + '_ {
        self.map.iter()
    }
    /// The iterator uses the Map to look up the entity so it can't out live the Map.
    pub fn get_neighbors(&self, coords: CubeCoordinate) -> impl Iterator<Item = Entity> + '_ {
        coords.get_neighbors().filter_map(|c| self.get(c))
    }
    /// The iterator uses the Map to look up the entity so it can't out live the Map. The iterator returns Entity and coordinates.
    pub fn get_neighbors_with_coords(
        &self,
        coords: CubeCoordinate,
    ) -> impl Iterator<Item = (Entity, CubeCoordinate)> + '_ {
        coords
            .get_neighbors()
            .filter_map(|c| self.get(c).map(|e| (e, c)))
    }

    /// Get RingIterator with minimum Radius of 1.
    /// The iterator uses the Map to look up the entity so it can't out live the Map.
    pub fn get_ring(
        &self,
        coords: CubeCoordinate,
        radius: usize,
    ) -> impl Iterator<Item = Entity> + '_ {
        coords.get_ring(radius).filter_map(|c| self.get(c))
    }
    /// Get NeighborhoodIterator with minimum Radius of 1.
    /// The iterator uses the Map to look up the entity so it can't out live the Map.
    pub fn get_area(
        &self,
        coords: CubeCoordinate,
        radius: usize,
    ) -> impl Iterator<Item = Entity> + '_ {
        coords.get_area(radius).filter_map(|c| self.get(c))
    }
    /// Get NeighborhoodIterator with minimum Radius of 1. The iterator returns Entity and coordinates.
    /// The iterator uses the Map to look up the entity so it can't out live the Map.
    pub fn get_area_with_coords(
        &self,
        coords: CubeCoordinate,
        radius: usize,
    ) -> impl Iterator<Item = (Entity, CubeCoordinate)> + '_ {
        coords
            .get_area(radius)
            .filter_map(|c| self.get(c).map(|e| (e, c)))
    }

    /// Uses depth search to collect a cluster of Tiles.
    /// Use filter to describe wich tiles should be clustered.
    pub fn depth_search_collect<F>(
        &self,
        coord: CubeCoordinate,
        filter: F,
    ) -> HashSet<CubeCoordinate>
    where
        F: Fn(CubeCoordinate, Entity) -> bool,
    {
        let mut stack: LinkedList<CubeCoordinate> = LinkedList::new();
        let mut out: HashSet<CubeCoordinate> = HashSet::new();
        stack.push_back(coord);
        out.insert(coord);
        while let Some(current) = stack.pop_back() {
            for neighbar_coord in current.get_neighbors() {
                if out.contains(&neighbar_coord) {
                    continue;
                }
                if let Some(entity) = self.get(neighbar_coord) {
                    if filter(neighbar_coord, entity) {
                        out.insert(neighbar_coord);
                        stack.push_back(neighbar_coord);
                    }
                }
            }
        }
        out
    }
}

// wrapper for tile specific data.
#[derive(Debug, Serialize, Deserialize, Clone)]
#[serde(tag = "type")]
pub enum AdditionalTileData {
    MarketInfo(MarketInfo),
    RgbData(RgbTileData),
    ChameleonData(ChameleonTileData),
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct TileSaveData {
    pub coord: CubeCoordinate,
    pub tile: TileType,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[serde(default)]
    pub data: Option<AdditionalTileData>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[serde(default)]
    pub reclaim: Option<f32>,
}

impl TileSaveData {
    pub fn new(coord: CubeCoordinate, tile: TileType) -> Self {
        Self {
            coord,
            tile,
            data: None,
            reclaim: None,
        }
    }
}

/// Struct to store Maps outside the esc system.
/// Save/load map:
/// ```ignore
/// let tiles: Query<ConsistentTileData, With<TileType>>;
/// let map = RawMap::save_game(tiles);
/// // other context
/// map.build_to_world(commands, without_scene);
/// // todo: send TilePlacedEvent to update resource calculation
/// ```
#[derive(Debug, Default, Clone, Serialize, Deserialize, Deref, DerefMut)]
pub struct RawMap(pub Vec<TileSaveData>);

impl RawMap {
    pub fn save_game(tiles: Query<ConsistentTileData, With<TileType>>) -> Self {
        let map = Vec::from_iter(tiles.iter().map(|data| TileSaveData {
            coord: *data.0,
            tile: *data.1,
            data: TileType::serialize_tile_data(data),
            reclaim: data.2.map(|f| f.timer.remaining_secs()),
        }));
        Self(map)
    }
    pub fn build_to_world(&self, mut commands: Commands, without_scene: bool) {
        for data in self.iter() {
            commands.spawn_empty().queue(
                AddTile::new(data.tile, data.coord)
                    .without_scene_cond(without_scene)
                    .with_data(data.data.clone()),
            );
        }
    }
}

#[derive(Debug, Default, Asset, TypePath, Serialize, Deserialize)]
pub struct GameWorld {
    pub map: RawMap,
    pub name: String,
}

#[derive(Debug, Error)]
#[non_exhaustive]
pub enum MapResourceLoadingError {
    #[error("Could not load world: {0}")]
    Io(#[from] std::io::Error),
    #[error("Could not parse utf8 file: {0}")]
    Utf8(#[from] std::string::FromUtf8Error),
    #[error("Could not parse Map data file: {0}")]
    Map(#[from] serde_json::Error),
}

#[derive(Default)]
pub struct SaveMapResourceLoader;
impl AssetLoader for SaveMapResourceLoader {
    type Asset = GameWorld;
    type Settings = ();
    type Error = MapResourceLoadingError;

    async fn load(
        &self,
        reader: &mut dyn bevy::asset::io::Reader,
        _settings: &(),
        _load_context: &mut bevy::asset::LoadContext<'_>,
    ) -> Result<Self::Asset, Self::Error> {
        let mut bytes = Vec::new();
        reader.read_to_end(&mut bytes).await?;
        let text = String::from_utf8(bytes)?;
        match serde_json::from_str(&text) {
            Ok(res) => Ok(res),
            Err(d) => Err(Self::Error::Map(d)),
        }
    }
}
