use std::{
    collections::{HashMap, VecDeque},
    hash::{DefaultHasher, Hasher},
    ops::AddAssign,
};

use bevy::prelude::*;
use fastrand::Rng;

use crate::random::TTERand;

use super::{
    gamemodes::{
        campaign::levels::{
            challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState, CampaignGroups,
        },
        challenge::ChallengeGameState,
        zen::ZenGameState,
        GameMode,
    },
    round::RoundCounter,
    tiles::TileType,
};

macro_rules! define_hand_plugin {
    ($({
        // in which game mode to run the initial hand generate system
        gamemode: $gamemode:expr,
        // in which substate of the game_mode to regenerate the hand
        gen_state: $state:expr$(,)?
    }),*) => {
        pub struct HandPlugin;
        impl Plugin for HandPlugin {
            fn build(&self, app: &mut App) {
                app.init_resource::<Hand>();
                app.init_resource::<Deck>();
                app.insert_resource(RandomCore::raw(0));

                $(
                    // reset the hand, so all slots are empty
                    // and generate a new hand
                    app.add_systems(
                        OnEnter($gamemode),
                        (reset_hand, reset_deck).chain(),
                    );
                    app.add_systems(
                        OnEnter($state),
                        (update_deck, generate_hand).chain(),
                    );
                )*
            }
        }
    }
}

define_hand_plugin![
    {
        gamemode: GameMode::Challenge,
        gen_state: ChallengeGameState::InRound,
    },
    {
        gamemode: CampaignGroups::CampaignChallenge,
        gen_state: CampaignChallengeState::InRound,
    },
    {
        gamemode: CampaignGroups::CampaignWettbewerb,
        gen_state: CampaignWettbewerbState::InRound,
    },
    {
        gamemode: GameMode::Zen,
        gen_state: ZenGameState::InRound,
    }
];

#[derive(Resource, Deref, DerefMut)]
/// When using in_game, pls consider system order, to keep the game deterministic.
// order log: before (<-), after (->)
// placer <- process_manual_redraw -> reward_timer_end
pub struct RandomCore(pub Rng);

impl RandomCore {
    fn raw(seed: u64) -> Self {
        Self(Rng::with_seed(seed))
    }
    pub fn change_seed(&mut self, seed: &str) {
        let mut hasher = DefaultHasher::new();
        hasher.write(seed.as_bytes());
        self.0 = Rng::with_seed(hasher.finish());
    }
}

/// how many cards fit into the inventory/hand
pub const HAND_SIZE: usize = 5;

#[derive(Resource)]
pub struct Hand {
    pub items: Vec<TileType>,
    pub selected: Option<usize>,
}

impl Default for Hand {
    fn default() -> Self {
        Self {
            items: Vec::with_capacity(HAND_SIZE),
            selected: Default::default(),
        }
    }
}

impl Hand {
    pub fn select_slot(&mut self, slot: usize) {
        // slot is always >=0 because of the type bound
        if slot < self.items.len() {
            self.selected = Some(slot);
        }
    }
    pub fn select_next_slot(&mut self) {
        if let Some(selection) = &mut self.selected {
            *selection += 1;
            if *selection >= self.items.len() {
                *selection = 0;
            }
        } else {
            self.select_slot(0);
        }
    }
    pub fn select_previous_slot(&mut self) {
        if let Some(selection) = &mut self.selected {
            // Set the slot idnex to the previous index
            // as this would cause a usize overflow if the hand is empty,
            // we have to use checked_sub and fall back to the last element.
            // If there are no elements this would also cause an overflow,
            // so we default to 0
            *selection = selection
                .checked_sub(1)
                .unwrap_or(self.items.len().saturating_sub(1));
        } else {
            // fall back to slot 0 if there are no items
            self.select_slot(self.items.len().saturating_sub(1));
        }
    }

    pub fn reset(&mut self) {
        // clear selection
        self.selected = None;
        // clear all hans slots
        self.items.clear();
    }
}

pub fn generate_hand(rng: ResMut<RandomCore>, hand: ResMut<Hand>, deck: ResMut<Deck>) {
    let hand = hand.into_inner();
    fill_hand(rng.into_inner(), hand, deck.into_inner());

    if hand.selected.is_none() {
        hand.selected = Some(0);
    }
}

pub fn fill_hand(rng: &mut RandomCore, hand: &mut Hand, deck: &mut Deck) {
    let cards_missing = HAND_SIZE - hand.items.len();
    // fill hand (and don't replace existing cards)
    let Ok(mut new_cards) = deck.draw(rng, cards_missing) else {
        warn!("No more cards left on deck");
        return;
    };

    hand.items.append(&mut new_cards);

    debug!("New Hand: {:?}", hand.items);
}

fn reset_hand(hand: ResMut<Hand>) {
    let hand = hand.into_inner();
    hand.reset();
}

pub fn reset_deck(mut deck: ResMut<Deck>) {
    // clear deck
    *deck = Deck::default();
}

pub fn update_deck(deck: ResMut<Deck>, round: Res<RoundCounter>) {
    let deck = deck.into_inner();

    deck.forced_queue.extend(TileType::forced_cards(**round));

    deck.add_assign(TileType::new_deck_in_round(round.0));
}

#[derive(Default, Resource)]
pub struct Deck {
    size: usize,
    forced_queue: VecDeque<TileType>,
    map: HashMap<TileType, usize>,
}

#[derive(PartialEq, Debug)]
pub struct NotEnoughCards;

impl Deck {
    pub fn draw(
        &mut self,
        rng: &mut RandomCore,
        mut draw_count: usize,
    ) -> Result<Vec<TileType>, NotEnoughCards> {
        if self.size < draw_count {
            return Err(NotEnoughCards);
        }

        let mut drawn_cards = Vec::with_capacity(draw_count);

        let forced_cards = self.forced_queue.len().min(draw_count);

        if forced_cards > 0 {
            drawn_cards.extend(self.forced_queue.drain(..forced_cards));
            draw_count -= forced_cards;
        }

        // Get a set of non distinct indices
        let mut choices = rng.choose_n(0..self.size, draw_count);
        rng.shuffle(&mut choices);

        self.size -= draw_count;

        for tile in TileType::list() {
            let Some(count) = self.map.get(&tile) else {
                continue;
            };
            choices = choices
                .into_iter()
                .filter_map(|choice| {
                    if choice >= *count {
                        //The choosen card is deeper in the pile
                        Some(choice - *count)
                    } else {
                        //The choose card is in this porition so add the type to the choosen cards and remove the choice
                        drawn_cards.push(tile);
                        None
                    }
                })
                .collect();
        }

        // Removing the cards while drawing isn't trivial and likely only doable if the choices are sorted so instead the cards get removed after all the choices are made
        for card in &drawn_cards {
            let card_count = self
                .map
                .get_mut(card)
                .expect("We choose the card so it has to be there.");
            *card_count -= 1;
            if *card_count == 0 {
                self.map.remove(card);
            }
        }

        Ok(drawn_cards)
    }
}

impl<D: Into<Deck> + Sized> std::ops::AddAssign<D> for Deck {
    fn add_assign(&mut self, rhs: D) {
        let rhs = rhs.into();
        self.size += rhs.size;
        for (tile, added) in rhs.map.into_iter() {
            if let Some(deck) = self.map.get_mut(&tile) {
                *deck += added;
            } else {
                self.map.insert(tile, added);
            }
        }
    }
}

impl<const N: usize> From<[(TileType, usize); N]> for Deck {
    fn from(list: [(TileType, usize); N]) -> Self {
        let mut size = 0;
        let list = list.into_iter().filter(|&(_, count)| {
            size += count;
            count != 0
        });
        let map = HashMap::from_iter(list);
        Self {
            size,
            map,
            forced_queue: VecDeque::new(),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_card_redraw_consistency() {
        let runs = 10;
        let draws = 3;
        let card_count = 5;
        let seed = 1234;
        let round = 0;

        let mut base_rng = RandomCore::raw(seed);
        let mut base_deck = Deck::default();
        base_deck.add_assign(TileType::new_deck_in_round(round));
        let baseline: Vec<Result<Vec<TileType>, NotEnoughCards>> = (0..draws)
            .map(|_| base_deck.draw(&mut base_rng, card_count))
            .collect();

        for _ in 0..runs {
            let mut comp_rng = RandomCore::raw(seed);
            let mut comp_deck = Deck::default();
            comp_deck.add_assign(TileType::new_deck_in_round(round));
            let comp: Vec<Result<Vec<TileType>, NotEnoughCards>> = (0..draws)
                .map(|_| comp_deck.draw(&mut comp_rng, card_count))
                .collect();
            assert_eq!(
                baseline, comp,
                "The deck should always return the same cards for the same seed"
            );
        }
    }
}
