use bevy::prelude::*;
use campaign::CampaignModePlugin;
use challenge::ChallengeModePlugin;
use creative::CreativeModePlugin;
use serde::{Deserialize, Serialize};
use zen::ZenModePlugin;

#[cfg(feature = "graphics")]
use crate::prelude::WithTranslationID;
use crate::{AppState, DisplayMode};

pub mod campaign;
pub mod challenge;
pub mod creative;
pub mod zen;

pub struct GameModePlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for GameModePlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_sub_state::<GameMode>();
        app.add_plugins(ChallengeModePlugin {
            display_mode: self.display_mode,
        });
        app.add_plugins(CreativeModePlugin {
            display_mode: self.display_mode,
        });
        app.add_plugins(CampaignModePlugin {
            display_mode: self.display_mode,
        });
        app.add_plugins(ZenModePlugin {
            display_mode: self.display_mode,
        });
    }
}

/// Which GameMode the player is playing
#[derive(SubStates, Default, Debug, Hash, PartialEq, Eq, Clone, Serialize, Deserialize)]
#[source(AppState = AppState::InGame)]
pub enum GameMode {
    /// default game mode
    /// where player has to meet the ever-growing resource target
    #[default]
    Challenge,
    /// creative game mode
    /// where the user has an infinite amount of tiles
    /// and can place tiles everywhere
    Creative,
    /// campaign game mode
    /// allows users to play through levels
    /// as a challenge or to learn how to play
    Campaign,
    /// zen game mode
    /// infinite play mode,
    /// where the round increases when all targets are reached
    Zen,
}
impl GameMode {
    pub fn list() -> Vec<GameMode> {
        vec![
            GameMode::Challenge,
            GameMode::Creative,
            GameMode::Campaign,
            GameMode::Zen,
        ]
    }
}

#[cfg(feature = "graphics")]
impl WithTranslationID for GameMode {
    fn get_translation_id(&self) -> &str {
        match self {
            GameMode::Challenge => "gamemode-challenge",
            GameMode::Creative => "gamemode-creative",
            GameMode::Campaign => "gamemode-campaign",
            GameMode::Zen => "gamemode-zen",
        }
    }
}

/// Marker trait
/// represents substates of GameMode
pub trait GameModeSubStates: SubStates {}
