use bevy::prelude::*;

use crate::{game::Successful, AppState, DisplayMode};

#[cfg(feature = "graphics")]
use crate::{
    game::time::pause_on_focus_loss,
    ui::in_game::{
        game_export::show_export_window,
        pause::{ui_pause_screen, PauseUiState},
    },
};

use super::{GameMode, GameModeSubStates};

/// number of slots in the creativemode inventory
#[allow(dead_code)]
pub const CREATIVE_INV_SLOT_COUNT: usize = 5;

pub struct CreativeModePlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for CreativeModePlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<CreativeGameState>();
        app.init_resource::<CreativeModeConfig>();

        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display_mode {
            use CreativeGameState::*;
            use GameMode::Creative;

            app.add_systems(
                Update,
                ui_pause_screen::<false>
                    .run_if(in_state(Creative))
                    .run_if(in_state(PauseUiState::Shown))
                    .before(show_export_window),
            );
            app.add_systems(Update, pause_on_focus_loss.run_if(in_state(InGame)));
        }
    }
}

#[derive(Debug, Clone, Copy, Default, PartialEq, Eq, Hash, SubStates)]
#[source(GameMode = GameMode::Creative)]
pub enum CreativeGameState {
    /// the player is currently able to interact with the world
    #[default]
    InGame,
}

impl GameModeSubStates for CreativeGameState {}

#[derive(Default, Resource)]
pub struct CreativeModeConfig {}

/// one-shot system to start a new game in creative mode
/// loads gamemode and configuration data from the GameConfig resource
/// ```ignore
/// world.run_system_once(init_creative_mode)
/// ```
pub fn init_creative_mode(
    _config: Res<CreativeModeConfig>,
    mut app_state: ResMut<NextState<AppState>>,
    mut gamemode: ResMut<NextState<GameMode>>,
    mut succ: ResMut<Successful>,
) {
    app_state.set(AppState::InGame);
    gamemode.set(GameMode::Creative);
    **succ = true;
}
