use bevy::prelude::*;
use serde::{Deserialize, Serialize};

use super::{GameMode, GameModeSubStates};
#[cfg(feature = "graphics")]
use crate::{
    game::time::pause_on_focus_loss,
    ui::in_game::{
        game_export::show_export_window,
        pause::{ui_pause_screen, PauseUiState},
    },
};
use crate::{
    game::{hand::RandomCore, round::RoundSystemSupport, Successful},
    AppState, Difficulty, DisplayMode,
};

/// debug placer slot count
#[cfg(debug_assertions)]
#[allow(dead_code)]
pub const DEBUG_PLACER_SIZE: usize = 1;

pub struct ChallengeModePlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for ChallengeModePlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<ChallengeGameState>();
        app.init_resource::<ChallengeModeConfig>();

        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display_mode {
            use ChallengeGameState::*;
            use GameMode::Challenge;

            app.add_systems(
                Update,
                ui_pause_screen::<true>
                    .run_if(in_state(Challenge))
                    .run_if(in_state(PauseUiState::Shown))
                    .before(show_export_window),
            );
            app.add_systems(Update, pause_on_focus_loss.run_if(in_state(InRound)));
        }
    }
}

#[derive(Debug, Clone, Copy, Default, PartialEq, Eq, Hash, SubStates)]
#[source(GameMode = GameMode::Challenge)]
pub enum ChallengeGameState {
    /// the player is currently in a round
    /// and can place tiles and redraw cards
    #[default]
    InRound,
    /// the player is between two rounds
    BetweenRounds,
    /// the player lost the game
    GameOver,
    /// Immutable game view
    /// the player can still use the camera to view the world
    Spectate,
}

impl GameModeSubStates for ChallengeGameState {}

impl RoundSystemSupport for ChallengeGameState {
    fn get_gameover_state() -> Self {
        Self::GameOver
    }
    fn get_phaseend_state() -> Self {
        Self::BetweenRounds
    }
    fn get_inround_state() -> Self {
        Self::InRound
    }
    fn get_spectate_state() -> Option<Self> {
        Some(Self::Spectate)
    }
}

#[derive(Debug, Clone, Default, Resource, Serialize, Deserialize)]
pub struct ChallengeModeConfig {
    pub seed: String,
    pub difficulty: Difficulty,
}

/// one-shot system to start a new game in challenge mode
/// loads gamemode and configuration data from the GameConfig resource
/// ```ignore
/// world.run_system_once(init_challenge_mode)
/// ```
pub fn init_challenge_mode(
    config: Res<ChallengeModeConfig>,
    mut seed: ResMut<RandomCore>,
    mut app_state: ResMut<NextState<AppState>>,
    mut gamemode: ResMut<NextState<GameMode>>,
    mut succ: ResMut<Successful>,
) {
    info!("Starting game with seed: {}", config.seed);
    seed.change_seed(&config.seed);
    app_state.set(AppState::InGame);
    gamemode.set(GameMode::Challenge);
    **succ = true;
}
