use bevy::prelude::*;
#[cfg(not(feature = "graphics"))]
use levels::CampaignChallenge;
#[cfg(feature = "graphics")]
use levels::CampaignTutorial;
use levels::{CampaignGroupWrapper, CampaignGroups, CampaignLevelsPlugin};
use traits::Level;

use crate::{
    game::{GameEnd, GamePauseState, Successful},
    DisplayMode,
};

#[cfg(feature = "graphics")]
use crate::ui::in_game::{
    game_export::show_export_window,
    pause::{ui_pause_screen, PauseUiState},
};

use super::GameMode;

pub mod traits;
#[macro_use]
pub mod macros;
pub mod levels;

pub struct CampaignModePlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for CampaignModePlugin {
    fn build(&self, app: &mut App) {
        // register config resource
        app.init_resource::<CampaignModeConfig>();
        app.add_sub_state::<CampaignLevelProgress>();

        app.add_event::<UpdateCampaignStorySegment>();
        // required because the ui code isn't macro generated
        // and thus cannot control the level state directly
        app.add_event::<CampaignLevelStart>();

        // register all groups and levels
        CampaignGroups::register(app);

        // plugin that adds all group & level plugins
        app.add_plugins(CampaignLevelsPlugin {
            display_mode: self.display_mode,
        });

        // add basic plugins required by every level
        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display_mode {
            app.add_systems(
                Update,
                ui_pause_screen::<false>
                    .run_if(in_state(CampaignLevelProgress::Running))
                    .run_if(in_state(PauseUiState::Shown))
                    .before(show_export_window),
            );
        }

        app.add_systems(OnEnter(CampaignLevelProgress::Outro), campaign_end);
    }
}

/// Possible error whilst parsing the campaign group
#[derive(Debug)]
pub enum CampaignGroupParserError {
    /// no group with this ID exists
    InvalidGroup,
    /// no level with the given id exists in the group
    InvalidLevel,
}

fn step_level_story<T: Level>(
    segment: Res<State<T>>,
    mut next_segment: ResMut<NextState<T>>,
    mut level_progress: ResMut<NextState<CampaignLevelProgress>>,
    mut gamepause: ResMut<NextState<GamePauseState>>,
) {
    if segment.get().get_progress() == CampaignLevelProgress::Intro {
        // continue the game time when leaving the intro screen
        gamepause.set(GamePauseState::Running);
    }
    if let Some(next) = segment.get().next() {
        let progress = next.get_progress();
        // pause the game on the outro screen
        if progress == CampaignLevelProgress::Outro {
            gamepause.set(GamePauseState::Paused);
        }
        info!("Level progress: {:?}", progress);
        level_progress.set(progress);
        next_segment.set(next);
    }
}

fn campaign_end(mut gameend: EventWriter<GameEnd>, mut succ: ResMut<Successful>) {
    gameend.send(GameEnd);
    **succ = true;
}

/// Story segment update event
/// containing the translation id of the segment
#[derive(Event, Debug, Deref)]
pub struct UpdateCampaignStorySegment(pub String);

/// Indicator state
/// consistency is maintained using the group and level macro
/// indicates which segment group the currently running level is in
#[derive(SubStates, Hash, PartialEq, Eq, Clone, Default, Debug)]
#[source(GameMode = GameMode::Campaign)]
pub enum CampaignLevelProgress {
    /// indicates that the level is in the start segment
    #[default]
    Intro,
    /// indicates that the level is in one of the middle parts
    Running,
    /// indicates that the level is completed an in the end segment
    Outro,
}

/// event that can be send to step from the first level segment
/// into the second one
/// received by the Level macro
#[derive(Event)]
pub struct CampaignLevelStart;

pub const LEVEL_TRANSLATION_ID_BASE: &str = "campaign-level";

/// broadcasts the new story segment state
fn display_story_segment<T: Level>(
    segment: Res<State<T>>,
    group: Res<State<CampaignGroups>>,
    mut ev: EventWriter<UpdateCampaignStorySegment>,
) {
    let segment_id = segment.get().get_id();
    let level_id = segment.get().get_level_id();
    let group_id = group.get().get_id();
    let translation_id = format!(
        "{}-{}-{}-{}",
        LEVEL_TRANSLATION_ID_BASE, group_id, level_id, segment_id
    );
    trace!("Broadcasting translations for {}", translation_id);
    ev.send(UpdateCampaignStorySegment(translation_id));
}

#[derive(Default, Resource)]
pub struct CampaignModeConfig {
    pub level: CampaignGroupWrapper,
}

impl Default for CampaignGroupWrapper {
    fn default() -> Self {
        #[cfg(feature = "graphics")]
        return Self::CampaignTutorial(CampaignTutorial::CameraTutorial);
        #[cfg(not(feature = "graphics"))]
        return Self::CampaignChallenge(CampaignChallenge::ChallengeEasy);
    }
}
