use bevy::{ecs::system::SystemParam, prelude::*};
use bevy_enum_filter::watch_for_enum;

use bot_placer::BotPlacerPlugin;
#[cfg(feature = "graphics")]
use cam::CamPlugin;
#[cfg(feature = "graphics")]
use creative_placer::{CreativeInventoryPlugin, CreativePlacerPlugin};
#[cfg(feature = "graphics")]
use keyboard_placer::KeyboardPlacerPlugin;
#[cfg(feature = "graphics")]
use placer::PlacerPlugin;

use crate::{coordinates::CubeCoordinate, DisplayMode};

use super::{
    build_area::BuildArea,
    hand::Hand,
    map::Map,
    metrics::{Metrics, MetricsRate},
    rewards::Reward,
    tiles::TileType,
    GameSimSet,
};

pub mod bot_placer;
#[cfg(feature = "graphics")]
pub mod cam;
#[cfg(feature = "graphics")]
pub mod creative_placer;
#[cfg(feature = "graphics")]
pub mod keyboard_placer;
#[cfg(feature = "graphics")]
pub mod placer;

pub struct ControlsPlugin {
    pub display_mode: DisplayMode,
}

impl Plugin for ControlsPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_event::<KeyboardEvent>();

        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display_mode {
            app.add_plugins(CamPlugin);
            app.add_plugins(PlacerPlugin);
            app.add_plugins(KeyboardPlacerPlugin);
            app.add_plugins(CreativeInventoryPlugin);
            app.add_plugins(CreativePlacerPlugin);
        }

        app.add_plugins(BotPlacerPlugin);

        app.configure_sets(
            Update,
            (PlacerSet::Placer, PlacerSet::UpdateMarker)
                .chain()
                .in_set(GameSimSet::Prepare),
        );

        // equivalent to: `app.add_enum_filter::<TileType>();
        // but needed for scheduling
        app.add_systems(
            Update,
            watch_for_enum::<TileType>.in_set(PlacerSet::UpdateMarker),
        );
    }
}

#[derive(SystemSet, Debug, Clone, PartialEq, Eq, Hash)]
pub enum PlacerSet {
    /// set should be used for all placers
    Placer,
    /// Set for `watch_for_enum`. Updates markers for specific TileType selection.
    UpdateMarker,
}

#[derive(Event)]
pub struct KeyboardEvent;

#[derive(SystemParam)]
pub struct GameData<'w, 's> {
    pub hand: ResMut<'w, Hand>,
    pub map: Res<'w, Map>,
    pub metrics: ResMut<'w, Metrics>,
    pub rate: Res<'w, MetricsRate>,
    pub build_area: Query<'w, 's, &'static BuildArea>,
    pub rewards: Query<'w, 's, (&'static Transform, &'static CubeCoordinate), With<Reward>>,
}
