use std::f32::consts::PI;

use crate::{
    coordinates::{CubeCoordinate, EDGE_LENGTH, TILE_SIZE},
    game::{
        build_area::BUILD_PLATE_HEIGHT,
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
                CampaignGroups,
            },
            challenge::ChallengeGameState,
            creative::CreativeGameState,
            zen::ZenGameState,
            GameMode,
        },
        hand::RandomCore,
        map::{Map, TileSaveData},
        placement_reclaim::{handle_tile_reclaim, PlacementTimer},
        resources::{TilePlacedEvent, TileReclaimedEvent},
        rewards::RewardReqEvent,
        tiles::{AddTile, RemoveTile, TileType},
        GamePauseState,
    },
    settings::ActiveSettingsBank,
    ui::{is_typing_egui, DebugMaterial},
};
use bevy::{
    pbr::{NotShadowCaster, NotShadowReceiver},
    prelude::*,
    render::{mesh::PrimitiveTopology, render_asset::RenderAssetUsages},
};
use bevy_egui::EguiContexts;

use super::{
    cam::PanOrbitCam,
    creative_placer::CreativeInventory,
    placer::{PlacerEvents, TileClickEvent},
    GameData, PlacerSet,
};

const TOP_COLOR: Color = Color::srgba(1.0, 1.0, 1.0, 0.0);
const BOTTOM_COLOR: Color = Color::srgba(1.0, 1.0, 1.0, 0.5);
const CURSOR_HEIGHT: f32 = 1.5;
const CURSOR_ORIGIN_HEIGHT: f32 = BUILD_PLATE_HEIGHT;

macro_rules! define_keyboard_placer_plugin {
    ($({
        // in which state to run the keyboard_placer
        state: $state:expr,
        // gamemode to handle
        gamemode: $gm:expr,

        click_system: $click_system:expr,
    }),*) => {

        pub struct KeyboardPlacerPlugin;

        impl Plugin for KeyboardPlacerPlugin{
            fn build(&self, app: &mut App) {
                use crate::prelude::never;

                app.add_systems(
                    Update,
                    move_cursor
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running))
                );

                $(
                    app.add_systems(OnEnter($gm), spawn_cursor);
                    app.add_systems(OnExit($gm), despawn_cursor);
                    app.add_systems(
                        Update,
                        $click_system.in_set(PlacerSet::Placer)
                            .run_if(in_state($state))
                            .run_if(in_state(GamePauseState::Running)),
                    );
                )*
            }
        }
    }
}

define_keyboard_placer_plugin![
    {
        state: ChallengeGameState::InRound,
        gamemode: GameMode::Challenge,
        click_system: cursor_click,
    },
    {
        state: CampaignChallengeState::InRound,
        gamemode: CampaignGroups::CampaignChallenge,
        click_system: cursor_click,
    },
    {
        state: CampaignWettbewerbState::InRound,
        gamemode: CampaignGroups::CampaignWettbewerb,
        click_system: cursor_click,
    },
    {
        state: CreativeGameState::InGame,
        gamemode: GameMode::Creative,
        click_system: creative_cursor_click,
    },
    {
        state: ZenGameState::InRound,
        gamemode: GameMode::Zen,
        click_system: cursor_click,
    }
];

#[derive(Component, Debug, Copy, Clone, Deref, DerefMut, Default)]
pub struct Cursor(pub CubeCoordinate);

fn spawn_cursor(
    mut commands: Commands,
    mut meshes: ResMut<Assets<Mesh>>,
    mut materials: ResMut<Assets<StandardMaterial>>,
) {
    let material = materials.add(StandardMaterial {
        base_color: Color::srgb(1.0, 1.0, 1.0),
        double_sided: true,
        cull_mode: None,
        alpha_mode: AlphaMode::Blend,
        depth_bias: 1000.0, // ensure cursor will be rendered after water
        unlit: true,
        ..default()
    });

    //outer vertecies of hexgon, slightly enlarged to prevent mesh flimmer
    let vertecies: Vec<Vec2> = vec![
        0.99 * Vec2::new(TILE_SIZE as f32 * 0.5, EDGE_LENGTH as f32 * 0.5),
        0.99 * Vec2::new(TILE_SIZE as f32 * 0.5, -EDGE_LENGTH as f32 * 0.5),
        0.99 * Vec2::new(0.0, -EDGE_LENGTH as f32),
        0.99 * Vec2::new(-TILE_SIZE as f32 * 0.5, -EDGE_LENGTH as f32 * 0.5),
        0.99 * Vec2::new(-TILE_SIZE as f32 * 0.5, EDGE_LENGTH as f32 * 0.5),
        0.99 * Vec2::new(0.0, EDGE_LENGTH as f32),
    ];
    let mut positions: Vec<[f32; 3]> = Vec::with_capacity(vertecies.len() * 6);
    let mut colors: Vec<[f32; 4]> = Vec::with_capacity(vertecies.len() * 6);

    let bottom_color = BOTTOM_COLOR.to_srgba().to_f32_array();
    let top_color = TOP_COLOR.to_srgba().to_f32_array();

    for i in 0..vertecies.len() {
        let a = vertecies[i];
        let b = vertecies[(i + 1) % vertecies.len()];
        positions.extend_from_slice(&[
            [a.x, CURSOR_ORIGIN_HEIGHT + 0.01, a.y],
            [b.x, CURSOR_ORIGIN_HEIGHT + 0.01, b.y],
            [a.x, CURSOR_HEIGHT, a.y],
            [a.x, CURSOR_HEIGHT, a.y],
            [b.x, CURSOR_ORIGIN_HEIGHT + 0.01, b.y],
            [b.x, CURSOR_HEIGHT, b.y],
        ]);
        colors.extend_from_slice(&[
            bottom_color,
            bottom_color,
            top_color,
            top_color,
            bottom_color,
            top_color,
        ]);
    }

    let mut mesh = Mesh::new(
        PrimitiveTopology::TriangleList,
        RenderAssetUsages::default(),
    );
    mesh.insert_attribute(Mesh::ATTRIBUTE_POSITION, positions);
    mesh.insert_attribute(Mesh::ATTRIBUTE_COLOR, colors);

    let mesh_h = meshes.add(mesh);

    commands.spawn((
        Mesh3d(mesh_h),
        MeshMaterial3d(material),
        Transform::default(),
        Visibility::Hidden,
        Cursor::default(),
        DebugMaterial("Cursor"),
        NotShadowCaster,
        NotShadowReceiver,
    ));
}

fn despawn_cursor(mut commands: Commands, cursor: Query<Entity, With<Cursor>>) {
    for c in cursor.iter() {
        commands.entity(c).despawn_recursive();
    }
}

fn move_cursor(
    mut cursor: Query<(&mut Transform, &mut Visibility, &mut Cursor), With<Cursor>>,
    cam_query: Query<&PanOrbitCam>,
    settings: Res<ActiveSettingsBank>,
    input: Res<ButtonInput<KeyCode>>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }

    const DEG60: f32 = 60.0 / 180.0 * PI;

    let cam = cam_query.single();
    let Ok((mut transform, mut visibility, mut cursor)) = cursor.get_single_mut() else {
        return;
    };

    let mut cursor_move = CubeCoordinate::default();

    if settings
        .keybindings
        .move_cursor_forward
        .any(|code| input.just_pressed(code))
    {
        cursor_move += CubeCoordinate::from(Vec2::from_angle(-cam.orbit.x));
    }
    if settings
        .keybindings
        .move_cursor_forward_right
        .any(|code| input.just_pressed(code))
    {
        cursor_move += CubeCoordinate::from(Vec2::from_angle(-cam.orbit.x + DEG60));
    }
    if settings
        .keybindings
        .move_cursor_backward_right
        .any(|code| input.just_pressed(code))
    {
        cursor_move += CubeCoordinate::from(Vec2::from_angle(-cam.orbit.x + DEG60 * 2.0));
    }
    if settings
        .keybindings
        .move_cursor_backward
        .any(|code| input.just_pressed(code))
    {
        cursor_move += CubeCoordinate::from(Vec2::from_angle(-cam.orbit.x + DEG60 * 3.0));
    }
    if settings
        .keybindings
        .move_cursor_backward_left
        .any(|code| input.just_pressed(code))
    {
        cursor_move += CubeCoordinate::from(Vec2::from_angle(-cam.orbit.x + DEG60 * 4.0));
    }
    if settings
        .keybindings
        .move_cursor_forward_left
        .any(|code| input.just_pressed(code))
    {
        cursor_move += CubeCoordinate::from(Vec2::from_angle(-cam.orbit.x + DEG60 * 5.0));
    }
    if cursor_move != CubeCoordinate::default() {
        *visibility = Visibility::Visible;
        **cursor += cursor_move;
        transform.translation = (**cursor).into();
    }
}

#[allow(clippy::too_many_arguments)]
/// cursor click function used to handle cursor clicks in game modes
/// that rely on a hand, debug placer and rewards
pub fn cursor_click(
    mut commands: Commands,
    mut events: PlacerEvents,
    mut game: GameData,
    cursor: Query<(&Visibility, &Cursor), With<Cursor>>,
    settings: Res<ActiveSettingsBank>,
    input: Res<ButtonInput<KeyCode>>,
    loading_tiles: Query<(&TileType, &CubeCoordinate), With<PlacementTimer>>,
    mut rng: ResMut<RandomCore>,
    debug_inv: Option<Res<CreativeInventory>>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }

    let Ok((visibility, cursor)) = cursor.get_single() else {
        return;
    };
    if Visibility::Hidden == visibility {
        return;
    }
    let Ok(build_area) = game.build_area.get_single() else {
        return;
    };
    if !settings
        .keybindings
        .cursor_click_left
        .any(|code| input.just_pressed(code))
    {
        return;
    }

    // the following reuses some code (but not all and changed!) may build generalized methode later.

    // try click rewards
    let mut interacted_with_rewards = false;
    for (_, coord) in game.rewards.iter() {
        if *coord != **cursor {
            continue;
        }
        // event will take care of deleting the reward
        events.reward_event.send(RewardReqEvent(*coord));
        interacted_with_rewards = true;
    }

    if interacted_with_rewards {
        return;
    }

    // try place tile
    let global_cursor = **cursor;

    let mut tile = None;
    let mut hand_index = None;

    if let Some(Some(st)) = debug_inv.and_then(|inv| inv.slots.get(inv.selected).cloned()) {
        tile = Some(st);
    } else if let Some(selected) = game.hand.selected {
        if let Some(st) = game.hand.items.get(selected) {
            tile = Some(*st);
            hand_index = game.hand.selected;
        }
    };
    let Some(tile) = tile else {
        // reclaim tile or send click event, if tile was hit
        if let Some(e) = game.map.get(global_cursor) {
            if handle_tile_reclaim(
                &mut commands,
                e,
                &loading_tiles,
                &mut game.hand,
                &mut game.metrics,
                &game.rate,
            ) {
                events
                    .ev_calc_reclaim
                    .send(TileReclaimedEvent(global_cursor));
                return;
            }
            events.tile_click_event.send(TileClickEvent {
                tile_pos: global_cursor,
            });
        }
        return;
    };

    // placer rules only apply to cards from the hand
    // debug placer tiles can always be placed anywhere
    if hand_index.is_some() {
        // check if global_cursor is inside the range of the build area
        if !build_area.check(global_cursor) {
            return;
        }

        // check if no tile is at that position already
        if let Some(e) = game.map.get(global_cursor) {
            if handle_tile_reclaim(
                &mut commands,
                e,
                &loading_tiles,
                &mut game.hand,
                &mut game.metrics,
                &game.rate,
            ) {
                events
                    .ev_calc_reclaim
                    .send(TileReclaimedEvent(global_cursor));
                return;
            }
            events.tile_click_event.send(TileClickEvent {
                tile_pos: global_cursor,
            });
            return;
        }

        // check if tile is adjacent to another tile (when at least one exists)
        if !game.map.is_empty() && game.map.get_neighbors(global_cursor).count() == 0 {
            return;
        }
    }

    commands
        .spawn_empty()
        .queue(AddTile::new(tile, global_cursor));

    {
        const RAND_GARBAGE_LENGTH: usize = 73;
        let mut random_garbage = [0u8; RAND_GARBAGE_LENGTH];
        let CubeCoordinate { q, r, s } = global_cursor;
        for coord in [q, r, s] {
            let count = (coord + rng.u32(..) as isize) as usize % RAND_GARBAGE_LENGTH;
            let window = &mut random_garbage[0..count];
            rng.fill(window);
        }
    }

    // tile was selected from hand
    // so we have to remove the card
    // as it has been used
    if let Some(index) = hand_index {
        game.hand.items.remove(index);
        if let Some(si) = game.hand.selected {
            // select last available slot,
            // if there is no card after the one we just placed
            if si >= game.hand.items.len() {
                game.hand.select_previous_slot();
            }
        }
    }

    // signal consumption
    events
        .ev_calc_place
        .send(TilePlacedEvent(TileSaveData::new(global_cursor, tile)));
}

#[allow(clippy::too_many_arguments)]
/// cursor click function used to handle cursor clicks in game modes
/// that rely on the creative inventory of a specific size
pub fn creative_cursor_click(
    mut commands: Commands,
    map: Res<Map>,
    cursor: Query<(&Visibility, &Cursor), With<Cursor>>,
    settings: Res<ActiveSettingsBank>,
    input: Res<ButtonInput<KeyCode>>,
    inv: Res<CreativeInventory>,
    mut ev_calc_place: EventWriter<TilePlacedEvent>,
    mut ev_calc_reclaim: EventWriter<TileReclaimedEvent>,
    mut ev_tile_click: EventWriter<TileClickEvent>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }
    let Ok((visibility, cursor)) = cursor.get_single() else {
        return;
    };
    if Visibility::Hidden == visibility {
        return;
    }

    let global_cursor = **cursor;

    if map.get(global_cursor).is_some() {
        if settings
            .keybindings
            .cursor_click_left
            .any(|code| input.just_pressed(code))
        {
            // show info on left click
            ev_tile_click.send(TileClickEvent {
                tile_pos: global_cursor,
            });
        } else if settings
            .keybindings
            .cursor_click_right
            .any(|code| input.just_pressed(code))
        {
            // remove on right click
            commands.queue(RemoveTile::new(global_cursor));
            ev_calc_reclaim.send(TileReclaimedEvent(global_cursor));
        }
    } else if settings
        .keybindings
        .cursor_click_left
        .any(|code| input.just_pressed(code))
    {
        // place tile on left click
        let Some(Some(tile)) = inv.slots.get(inv.selected) else {
            // no slot selected or slot empty
            return;
        };
        commands
            .spawn_empty()
            .queue(AddTile::new(*tile, global_cursor));

        // signal consumption
        ev_calc_place.send(TilePlacedEvent(TileSaveData::new(global_cursor, *tile)));
    }
}
