use std::f32::consts::PI;

use bevy::{core_pipeline::tonemapping::Tonemapping, input::mouse::MouseWheel, prelude::*};
use bevy_egui::EguiContexts;

use crate::{
    game::{
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, tutorial::camera::CameraTutorialStates,
                wettbewerb::CampaignWettbewerbState, CampaignGroups, CampaignTutorial,
            },
            challenge::ChallengeGameState,
            creative::CreativeGameState,
            zen::ZenGameState,
            GameMode,
        },
        GamePauseState,
    },
    prelude::{in_state_any, Cast},
    settings::ActiveSettingsBank,
    ui::{is_typing_egui, is_using_egui, theme::BACKGROUND_COLOR},
};

const LIMIT_RADIUS_MIN: f32 = 1.0;
const LIMIT_RADIUS_MAX: f32 = 200.0;
const LIMIT_Y_MAX: f32 = PI / 2.0 * 0.9;
const LIMIT_Y_MIN: f32 = PI / 2.0 * 0.0;
const SOFT_ELIMIT: f32 = PI / 2.0 * 0.2;

/// PAN_SPEED per radius
const KEYBOARD_PAN_SPEED: f32 = 1.0;
const KEYBOARD_ORBIT_SPEED: f32 = PI / 2.0;
const KEYBOARD_ZOOM_SPEED: f32 = 20.0;

const SCROLL_ZOOM_FACTOR: f32 = if cfg!(any(target_arch = "wasm32", target_arch = "wasm64")) {
    0.002
} else {
    0.2
};

macro_rules! define_camera_plugin {
    ($({
        states: $states:expr,
        gamemode: $gm:expr$(,)?
    }),*) => {
        pub struct CamPlugin;
        impl Plugin for CamPlugin {
            fn build(&self, app: &mut App) {
                use crate::prelude::never;

                app.insert_resource(InputInfo::new());

                $(
                    app.add_systems(
                        OnEnter($gm),
                        (setup_cam, rotate_cam).chain(),
                    );
                    app.add_systems(OnExit($gm), despawn_cam);
                )*
                app.add_systems(
                    Update,
                    ((
                        (
                            handle_mouse_input,
                            handle_scoll_input,
                            handle_touch_input,
                            handle_keyboard_input,
                        ),
                        rotate_cam,
                    )
                    .chain())
                    .run_if(never()$(.or(in_state_any($states)))*)
                    .run_if(in_state(GamePauseState::Running)),
                );
                app.add_systems(
                    Update,
                    update_cam_settings.run_if(resource_changed::<ActiveSettingsBank>),
                );
            }
        }
    };
}

define_camera_plugin![
    {
        states: vec![ChallengeGameState::InRound, ChallengeGameState::Spectate],
        gamemode: GameMode::Challenge
    },
    {
        states: vec![CreativeGameState::InGame],
        gamemode: GameMode::Creative
    },
    {
        states: vec![CameraTutorialStates::InGame],
        gamemode: CampaignTutorial::CameraTutorial,
    },
    {
        states: vec![CampaignChallengeState::InRound, CampaignChallengeState::Spectate],
        gamemode: CampaignGroups::CampaignChallenge,
    },
    {
        states: vec![CampaignWettbewerbState::InRound, CampaignWettbewerbState::Spectate],
        gamemode: CampaignGroups::CampaignWettbewerb,
    },
    {
        states: vec![ZenGameState::InRound],
        gamemode: GameMode::Zen
    }
];

#[derive(Component, Clone, Copy)]
pub struct PanOrbitCam {
    pub focus: Vec3,
    pub orbit: Vec2,
    pub radius: f32,
}
impl Default for PanOrbitCam {
    fn default() -> Self {
        Self {
            focus: Vec3::new(0.0, 0.0, 0.0),
            orbit: Vec2::ZERO,
            radius: 15.0,
        }
    }
}

#[derive(Resource)]
pub struct InputInfo {
    pub moved: Vec2,
    pub touch_placeable: bool,
    pub touch_down: f64,
}

impl Default for InputInfo {
    fn default() -> Self {
        Self::new()
    }
}

impl InputInfo {
    pub fn new() -> Self {
        Self {
            moved: Vec2::new(0.0, 0.0),
            touch_placeable: true,
            touch_down: 0.0,
        }
    }
}

fn setup_cam(mut commands: Commands, settings: Res<ActiveSettingsBank>) {
    #[cfg(target_os = "android")]
    let msaa = Msaa::Off;
    #[cfg(not(target_os = "android"))]
    let msaa = Msaa::default();

    commands.spawn((
        Transform::default(),
        Camera3d::default(),
        msaa,
        Camera {
            hdr: settings.graphics.hdr,
            ..default()
        },
        if settings.graphics.hdr {
            Tonemapping::default()
        } else {
            Tonemapping::None
        },
        DistanceFog {
            color: BACKGROUND_COLOR.cast(),
            falloff: FogFalloff::Linear {
                start: 8.0 * 10.0 * 5.0 * 0.8,
                end: 8.0 * 10.0 * 5.0,
            },
            ..default()
        },
        PanOrbitCam::default(),
    ));
}

fn update_cam_settings(
    settings: Res<ActiveSettingsBank>,
    mut cams: Query<(&mut Camera, &mut Tonemapping), With<PanOrbitCam>>,
) {
    for (mut cam, mut tonemapping) in cams.iter_mut() {
        cam.hdr = settings.graphics.hdr;
        *tonemapping = if settings.graphics.hdr {
            Tonemapping::default()
        } else {
            Tonemapping::None
        };
    }
}

fn despawn_cam(mut commands: Commands, cams: Query<Entity, With<PanOrbitCam>>) {
    for cam in cams.iter() {
        commands.entity(cam).despawn_recursive();
    }
}

/// Takes scroll Event on changes Cam Radius
fn handle_scoll_input(
    mut contexts: EguiContexts,
    mut ev_scroll: EventReader<MouseWheel>,
    mut cam_query: Query<&mut PanOrbitCam>,
) {
    if is_using_egui(&mut contexts) {
        return;
    }
    let mut cam = cam_query.single_mut();
    let mut scroll = 0.0;
    for ev in ev_scroll.read() {
        scroll += ev.y;
    }
    if scroll.abs() > 0.0 {
        cam.radius -= scroll * cam.radius * SCROLL_ZOOM_FACTOR;

        cam.radius = cam.radius.clamp(LIMIT_RADIUS_MIN, LIMIT_RADIUS_MAX);
    }
}

/// Takes mouse input and calculates Cam position.
pub fn handle_mouse_input(
    mut contexts: EguiContexts,
    windows: Query<&Window>,
    mut cam_query: Query<(&mut PanOrbitCam, &Projection)>,
    mut ev_motion: EventReader<CursorMoved>,
    settings: Res<ActiveSettingsBank>,
    input_mouse: Res<ButtonInput<MouseButton>>,
    mut mouse_info: ResMut<InputInfo>,
) {
    let orbit_button = settings.mouse.orbit;
    let pan_button = settings.mouse.pan;

    if !input_mouse.pressed(pan_button) && !input_mouse.pressed(orbit_button) {
        mouse_info.moved = Vec2::new(0.0, 0.0);
        return;
    }

    if is_using_egui(&mut contexts) {
        return;
    }

    let (mut cam, projection) = cam_query.single_mut();

    let mut mouse_delta = Vec2::ZERO;
    for ev in ev_motion.read() {
        if let Some(delta) = ev.delta {
            mouse_delta += delta;
        }
    }

    mouse_info.moved += mouse_delta;

    let Ok(window) = windows.get_single() else {
        return;
    };
    let window = Vec2::new(window.width(), window.height());

    if input_mouse.pressed(pan_button) {
        let mut pan = mouse_delta;
        if let Projection::Perspective(projection) = projection {
            pan *= Vec2::new(projection.fov * projection.aspect_ratio, projection.fov) / window;
            pan *= cam.radius;
        }
        let yaw = cam.orbit.x;
        cam.focus -= Vec3 {
            x: pan.x * yaw.sin() - pan.y * yaw.cos(),
            y: 0.0,
            z: pan.y * yaw.sin() + pan.x * yaw.cos(),
        };
    } else if input_mouse.pressed(orbit_button) {
        let mut orbit = -mouse_delta / window * PI * 2.0;

        if orbit.y > 0.0 {
            let max = (LIMIT_Y_MAX - cam.orbit.y) / SOFT_ELIMIT;
            if max < 1.0 {
                orbit.y *= max.powi(2);
            }
        } else if orbit.y < 0.0 {
            let min = (cam.orbit.y - LIMIT_Y_MIN) / SOFT_ELIMIT;
            if min < 1.0 {
                orbit.y *= min.powi(2);
            }
        }

        cam.orbit += orbit;
        while cam.orbit.x >= 2.0 * PI {
            cam.orbit.x -= 2.0 * PI;
        }
        while cam.orbit.x < 0.0 {
            cam.orbit.x += 2.0 * PI;
        }
        cam.orbit.y = cam.orbit.y.clamp(LIMIT_Y_MIN, LIMIT_Y_MAX);
    }
}

pub fn handle_touch_input(
    mut contexts: EguiContexts,
    windows: Query<&Window>,
    touches: Res<Touches>,
    mut mouse_info: ResMut<InputInfo>,
    mut cam_query: Query<(&mut PanOrbitCam, &Projection)>,
    touch_evr: EventReader<TouchInput>,
    time: Res<Time>,
) {
    let fingercount = touches.iter().count();

    if fingercount == 0 {
        mouse_info.touch_placeable = true;
        mouse_info.touch_down = time.elapsed_secs_f64();
        return;
    }

    if fingercount > 1 {
        mouse_info.touch_placeable = false;
    }

    if is_using_egui(&mut contexts) {
        return;
    }

    if touch_evr.is_empty() {
        return;
    }

    let Ok(window) = windows.get_single() else {
        return;
    };
    let window = Vec2::new(window.width(), window.height());

    let (mut cam, projection) = cam_query.single_mut();

    let mut center = Vec2::ZERO;
    let mut prev_center = Vec2::ZERO;
    for finger in touches.iter() {
        center += finger.position();
        prev_center += finger.previous_position();
    }
    center /= fingercount as f32;
    prev_center /= fingercount as f32;

    if fingercount == 1 {
        // pan - calculate average finger movement
        let mut pan = center - prev_center;
        if let Projection::Perspective(projection) = projection {
            pan *= Vec2::new(projection.fov * projection.aspect_ratio, projection.fov) / window;
            pan *= cam.radius;
        }
        //pan *= 1.5; // abetrare increase
        let yaw = cam.orbit.x;
        cam.focus -= Vec3 {
            x: pan.x * yaw.sin() - pan.y * yaw.cos(),
            y: 0.0,
            z: pan.y * yaw.sin() + pan.x * yaw.cos(),
        };
    } else {
        // orbit
        let mut orbit = -(center - prev_center) / window * PI * 2.0;

        // use avg. finger rotation around center for rotation
        orbit.x += touches.iter().fold(0.0, |acc, x| {
            if (x.previous_position() - prev_center).length_squared() > 0.1
                && (x.position() - center).length_squared() > 0.1
            {
                acc + ((x.previous_position() - prev_center).angle_to(x.position() - center))
            } else {
                0.0
            }
        }) / fingercount as f32;

        if orbit.y > 0.0 {
            let max = (LIMIT_Y_MAX - cam.orbit.y) / SOFT_ELIMIT;
            if max < 1.0 {
                orbit.y *= max.powi(2);
            }
        } else if orbit.y < 0.0 {
            let min = (cam.orbit.y - LIMIT_Y_MIN) / SOFT_ELIMIT;
            if min < 1.0 {
                orbit.y *= min.powi(2);
            }
        }

        cam.orbit += orbit;
        while cam.orbit.x >= 2.0 * PI {
            cam.orbit.x -= 2.0 * PI;
        }
        while cam.orbit.x < 0.0 {
            cam.orbit.x += 2.0 * PI;
        }
        cam.orbit.y = cam.orbit.y.clamp(LIMIT_Y_MIN, LIMIT_Y_MAX);

        // zoom - use avg. finger distance quotient
        let mut zoom = touches.iter().fold(0.0, |acc, x| {
            acc + (x.previous_position() - prev_center).length() / (x.position() - center).length()
        });
        zoom /= fingercount as f32;
        cam.radius *= zoom.powi(2); // abetrare increase
        cam.radius = cam.radius.clamp(LIMIT_RADIUS_MIN, LIMIT_RADIUS_MAX);
    }
}

fn handle_keyboard_input(
    mut cam_query: Query<&mut PanOrbitCam>,
    settings: Res<ActiveSettingsBank>,
    input: Res<ButtonInput<KeyCode>>,
    time: Res<Time>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }

    let mut cam = cam_query.single_mut();

    if settings
        .keybindings
        .top_down_view
        .any(|code| input.just_pressed(code))
    {
        cam.orbit = Vec2::ZERO;
    }

    let mut pan = Vec2::ZERO;

    if settings.keybindings.pan_top.any(|code| input.pressed(code)) {
        pan -= Vec2::Y;
    }
    if settings.keybindings.pan_bot.any(|code| input.pressed(code)) {
        pan += Vec2::Y;
    }
    if settings
        .keybindings
        .pan_left
        .any(|code| input.pressed(code))
    {
        pan -= Vec2::X;
    }
    if settings
        .keybindings
        .pan_right
        .any(|code| input.pressed(code))
    {
        pan += Vec2::X;
    }

    if pan != Vec2::ZERO {
        pan *= time.delta_secs();
        pan *= cam.radius;
        pan *= KEYBOARD_PAN_SPEED;
        let yaw = cam.orbit.x;
        cam.focus += Vec3 {
            x: pan.x * yaw.sin() - pan.y * yaw.cos(),
            y: 0.0,
            z: pan.y * yaw.sin() + pan.x * yaw.cos(),
        };
    }
    let mut orbit = Vec2::ZERO;
    if settings
        .keybindings
        .orbit_top
        .any(|code| input.pressed(code))
    {
        orbit -= Vec2::Y;
    }
    if settings
        .keybindings
        .orbit_bot
        .any(|code| input.pressed(code))
    {
        orbit += Vec2::Y;
    }
    if settings
        .keybindings
        .orbit_left
        .any(|code| input.pressed(code))
    {
        orbit -= Vec2::X;
    }
    if settings
        .keybindings
        .orbit_right
        .any(|code| input.pressed(code))
    {
        orbit += Vec2::X;
    }

    if orbit != Vec2::ZERO {
        orbit *= KEYBOARD_ORBIT_SPEED;
        orbit *= time.delta_secs();

        if orbit.y > 0.0 {
            let max = (LIMIT_Y_MAX - cam.orbit.y) / SOFT_ELIMIT;
            if max < 1.0 {
                orbit.y *= max.powi(2);
            }
        } else if orbit.y < 0.0 {
            let min = (cam.orbit.y - LIMIT_Y_MIN) / SOFT_ELIMIT;
            if min < 1.0 {
                orbit.y *= min.powi(2);
            }
        }

        cam.orbit += orbit;
        while cam.orbit.x >= 2.0 * PI {
            cam.orbit.x -= 2.0 * PI;
        }
        while cam.orbit.x < 0.0 {
            cam.orbit.x += 2.0 * PI;
        }
        cam.orbit.y = cam.orbit.y.clamp(LIMIT_Y_MIN, LIMIT_Y_MAX);
    }

    let mut zoom = 0.0;
    if settings.keybindings.zoom_in.any(|code| input.pressed(code)) {
        zoom -= 1.0;
    }
    if settings
        .keybindings
        .zoom_out
        .any(|code| input.pressed(code))
    {
        zoom += 1.0;
    }
    if zoom != 0.0 {
        zoom *= KEYBOARD_ZOOM_SPEED;
        zoom *= time.delta_secs();
        cam.radius = (cam.radius + zoom).clamp(LIMIT_RADIUS_MIN, LIMIT_RADIUS_MAX);
    }
}

/// Takes Cam3DMan data and moves the game Cam.
pub fn rotate_cam(mut cam_query: Query<(&PanOrbitCam, &mut Transform), Changed<PanOrbitCam>>) {
    let Some((cam, mut cam_transform)) = cam_query.iter_mut().next() else {
        return;
    };
    cam_transform.translation = cam.focus;
    cam_transform.translation.y += 0.2;
    cam_transform.rotation = Quat::from_xyzw(-0.5, -0.5, -0.5, 0.5);
    cam_transform.rotate_y(cam.orbit.x);
    cam_transform.rotate_local_x(cam.orbit.y);

    let rot_vec = cam_transform.forward();
    cam_transform.translation -= rot_vec * cam.radius;
}
