use std::collections::HashMap;

use bevy::prelude::*;

use crate::{
    coordinates::CubeCoordinate,
    game::{
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
            },
            challenge::ChallengeGameState,
            creative::CreativeGameState,
            zen::ZenGameState,
        },
        hand::RandomCore,
        map::TileSaveData,
        placement_reclaim::{handle_tile_reclaim, PlacementTimer},
        report::Warnings,
        resources::{TilePlacedEvent, TileReclaimedEvent},
        tiles::{AddTile, RemoveTile, TileType},
        GamePauseState,
    },
    DisplayMode,
};

use super::{GameData, PlacerSet};

macro_rules! define_bot_placer_plugin {
    ($({
        // state in which placer should be generated
        state: $state:expr,
        // is placer restricted
        restricted: $restricted:expr,
    }),*) => {
        pub struct BotPlacerPlugin;
        impl Plugin for BotPlacerPlugin {
            fn build(&self, app: &mut App) {
                app.add_event::<TileActionEvent>();
                $(
                    app.add_systems(
                        Update,
                        bot_placer::<$restricted>
                            .in_set(PlacerSet::Placer)
                            .run_if(in_state($state))
                            .run_if(in_state(GamePauseState::Running)),
                    );
                )*
            }
        }
    }
}
define_bot_placer_plugin![
    {
        state: ChallengeGameState::InRound,
        restricted: true,
    },
    {
        state: CreativeGameState::InGame,
        restricted: false,
    },
    {
        state: CampaignChallengeState::InRound,
        restricted: true,
    },
    {
        state: CampaignWettbewerbState::InRound,
        restricted: true,
    },
    {
        state: ZenGameState::InRound,
        restricted: true,
    }
];

#[derive(Debug, Clone)]
pub enum TileActionType {
    Place(TileSaveData),
    Take(CubeCoordinate),
}

#[derive(Debug, Clone, Event, Deref, DerefMut)]
pub struct TileActionEvent(pub TileActionType);

/// Enum to track multiple actions on one CubeCoordinate in a single tick.
enum InTickAction {
    Placed,
    Taken,
}

#[allow(clippy::too_many_arguments)]
pub fn bot_placer<const RESTRICTED: bool>(
    mut commands: Commands,
    mut ev_calc_place: EventWriter<TilePlacedEvent>,
    mut ev_calc_reclaim: EventWriter<TileReclaimedEvent>,
    mut game: GameData,
    loading_tiles: Query<(&TileType, &CubeCoordinate), With<PlacementTimer>>,
    mut rng: ResMut<RandomCore>,
    mut actions: EventReader<TileActionEvent>,
    mode: Res<State<DisplayMode>>,
    mut warnings: EventWriter<Warnings>,
) {
    let build_area = game.build_area.get_single();
    if RESTRICTED && build_area.is_err() {
        return;
    }

    let mut interacted: HashMap<CubeCoordinate, InTickAction> = HashMap::new();
    for action in actions.read() {
        match &action.0 {
            TileActionType::Place(place) => {
                let mut hand_index = None;
                if RESTRICTED {
                    // check if tile was interacted with in this frame
                    if let Some(InTickAction::Placed) = interacted.get(&place.coord) {
                        continue;
                    }

                    // check if coord is inside the range of the build area
                    if !build_area
                        .as_ref()
                        .expect("None was checked before.")
                        .check(place.coord)
                    {
                        warnings.send(Warnings::place());
                        warn!("Tried placing tile outside of the build area.");
                        continue;
                    }

                    // check if no tile is at that position already
                    if game.map.get(place.coord).is_some() {
                        warn!("Tried placing tile on occupied place.");
                        warnings.send(Warnings::place());
                        continue;
                    }

                    // check if tile is adjacent to another tile (when at least one exists)
                    if !game.map.is_empty() && game.map.get_neighbors(place.coord).count() == 0 {
                        warn!("Tried placing isolated tile.");
                        warnings.send(Warnings::place());
                        continue;
                    }

                    // check if tile is in the hand.
                    let Some(index) = game.hand.items.iter().position(|&r| r == place.tile) else {
                        warn!("Tried placing tile not on hand: {:?}", place.tile);
                        warnings.send(Warnings::place());
                        continue;
                    };
                    hand_index = Some(index);
                }

                let mut add_tile = AddTile::new(place.tile, place.coord);

                if *mode.get() == DisplayMode::Headless {
                    add_tile = add_tile.without_scene();
                }

                add_tile = add_tile.with_data(place.data.clone());

                if RESTRICTED {
                    interacted.insert(place.coord, InTickAction::Placed);
                }

                commands.spawn_empty().queue(add_tile);

                {
                    const RAND_GARBAGE_LENGTH: usize = 73;
                    let mut random_garbage = [0u8; RAND_GARBAGE_LENGTH];
                    let CubeCoordinate { q, r, s } = place.coord;
                    for coord in [q, r, s] {
                        let count = (coord + rng.u32(..) as isize) as usize % RAND_GARBAGE_LENGTH;
                        let window = &mut random_garbage[0..count];
                        rng.fill(window);
                    }
                }

                // tile was selected from hand
                // so we have to remove the card
                // as it has been used
                if let Some(index) = hand_index {
                    game.hand.items.remove(index);
                    if let Some(si) = game.hand.selected {
                        // select last available slot,
                        // if there is no card after the one we just placed
                        if si >= game.hand.items.len() {
                            game.hand.select_previous_slot();
                        }
                    }
                }

                // signal consumption
                ev_calc_place.send(TilePlacedEvent(TileSaveData::new(place.coord, place.tile)));
            }
            TileActionType::Take(coord) => {
                if RESTRICTED {
                    if let Some(InTickAction::Taken) = interacted.get(coord) {
                        continue;
                    }
                    if let Some(e) = game.map.get(*coord) {
                        if handle_tile_reclaim(
                            &mut commands,
                            e,
                            &loading_tiles,
                            &mut game.hand,
                            &mut game.metrics,
                            &game.rate,
                        ) {
                            ev_calc_reclaim.send(TileReclaimedEvent(*coord));
                            interacted.insert(*coord, InTickAction::Taken);
                        } else {
                            warn!("Reclaiming tile not possible.");
                            warnings.send(Warnings::take());
                        }
                    }
                } else {
                    #[allow(clippy::collapsible_else_if)]
                    if game.map.get(*coord).is_some() {
                        commands.queue(RemoveTile::new(*coord));
                        ev_calc_reclaim.send(TileReclaimedEvent(*coord));
                    }
                }
            }
        }
    }
}
