pub mod challenge_bot;
pub mod creative_bot;
pub mod data;

#[cfg(feature = "graphics")]
use crate::prelude::WithTranslationID;
use crate::{
    game::{
        gamemodes::{campaign::levels::init_campaign_mode, zen::init_zen_mode},
        rewards::RewardReqEvent,
    },
    websocket::{WebSocket, WebSocketConnector, WsError},
    AppState, DisplayMode, GameConfig,
};
use bevy::{
    app::Plugin,
    ecs::system::{RunSystemOnce, SystemParam},
    prelude::*,
    utils::HashSet,
};
use challenge_bot::ChallengeBotPlugin;
use creative_bot::{CreativeBotPlugin, TileStatusReq};
use data::{ActionList, BotAction, BotEvent, BotEventKind, NotInGame, NotInGameResponse};

use super::{
    controls::bot_placer::{TileActionEvent, TileActionType},
    gamemodes::{
        campaign::{levels::CampaignBotType, CampaignModeConfig},
        challenge::init_challenge_mode,
        creative::init_creative_mode,
        GameMode,
    },
    resources::ConfigureMarketReq,
    round::ManualRedrawEvent,
    tiles::{chameleon_tile::ChangeChameleonTile, rgb_tile::ChangeRgbTileColor},
    GameSimSet,
};

/// connecting timeout (timer gets reset on every connection step).
const CONNECTING_TIMEOUT: f32 = 1.0;
/// Limits the amount of action_lists received in one tick. The rest will be handled in the next tick.
const MAX_ACTION_LISTS_PER_TICK: usize = 100;

pub struct BotPlugin {
    pub mode: DisplayMode,
}
impl Plugin for BotPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_systems(
            Update,
            handle_connecting.run_if(resource_exists::<BotConnecting>),
        );
        app.add_systems(
            Update,
            in_game_handler
                .in_set(GameSimSet::Receive)
                .run_if(resource_exists::<BotConnection>)
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(
            Update,
            not_in_game_handler
                .before(in_game_handler)
                .run_if(resource_exists::<BotConnection>)
                .run_if(not(in_state(AppState::InGame)))
                .run_if(|conn_state: Res<BotConnectingState>| {
                    BotConnectingState::None != *conn_state
                }),
        );
        app.add_systems(
            Update,
            timeout_bot_connection
                .run_if(resource_exists::<BotConnecting>.or(resource_exists::<BotConnection>))
                .run_if(not(in_state(AppState::InGame))),
        );
        app.add_systems(OnExit(AppState::InGame), clean_up);
        if let DisplayMode::Headless = self.mode {
            app.add_systems(Update, headless_housekeeping);
        }

        app.init_resource::<BotSettings>();
        app.init_resource::<BotState>();
        app.init_resource::<BotConnectingState>();
        app.init_resource::<ConnectingTimeout>();
        app.add_event::<ActionEvent>();
        app.add_event::<ConnectionLostEvent>();
        app.add_event::<BotDataErrorEvent>();
        app.add_event::<BotEvent>();
        app.add_event::<BotActionOverflowEvent>();
        app.add_event::<BotTimeoutEvent>();

        // bot mode handlers

        app.add_plugins(ChallengeBotPlugin);
        app.add_plugins(CreativeBotPlugin { mode: self.mode });
    }
}

#[derive(Debug, Resource, Deref, DerefMut)]
pub struct ConnectingTimeout(pub Timer);

impl Default for ConnectingTimeout {
    fn default() -> Self {
        Self(Timer::from_seconds(CONNECTING_TIMEOUT, TimerMode::Once))
    }
}

fn timeout_bot_connection(
    mut commands: Commands,
    time: Res<Time<Real>>,
    mut timer: ResMut<ConnectingTimeout>,
    mut state: ResMut<BotConnectingState>,
) {
    if !state.is_waiting_state() {
        return;
    }
    timer.tick(time.delta());
    if timer.just_finished() {
        commands.remove_resource::<BotConnecting>();
        commands.remove_resource::<BotConnection>();
        *state = BotConnectingState::Timeouted;
        warn!("Connection to Bot timeout.");
    }
}

fn clean_up(
    mut commands: Commands,
    mut timer: ResMut<ConnectingTimeout>,
    mut state: ResMut<BotConnectingState>,
) {
    timer.reset();
    commands.remove_resource::<BotConnecting>();
    commands.remove_resource::<BotConnection>();
    *state = BotConnectingState::None;
}

#[derive(Resource, Default, Eq, PartialEq)]
pub enum BotConnectingState {
    #[default]
    None,
    // connecting
    WaitingWs,
    WaitingDiscovery,
    WaitingConn,
    // failed
    ConnFailed,
    UrlError,
    ModeMismatch,
    Timeouted,
    Rejected,
    Incompatible,
}
impl BotConnectingState {
    pub fn is_waiting_state(&self) -> bool {
        #[allow(clippy::match_like_matches_macro)]
        match self {
            BotConnectingState::WaitingWs
            | BotConnectingState::WaitingDiscovery
            | BotConnectingState::WaitingConn => true,
            _ => false,
        }
    }
    pub fn is_error_state(&self) -> bool {
        #[allow(clippy::match_like_matches_macro)]
        match self {
            BotConnectingState::None
            | BotConnectingState::WaitingWs
            | BotConnectingState::WaitingDiscovery
            | BotConnectingState::WaitingConn => false,
            _ => true,
        }
    }
}

#[cfg(feature = "graphics")]
impl WithTranslationID for BotConnectingState {
    fn get_translation_id(&self) -> &str {
        match self {
            BotConnectingState::None => "bot-state-none",
            BotConnectingState::WaitingWs => "bot-state-waiting",
            BotConnectingState::WaitingDiscovery => "bot-state-waiting",
            BotConnectingState::WaitingConn => "bot-state-waiting",
            BotConnectingState::ConnFailed => "bot-state-conn_failed",
            BotConnectingState::UrlError => "bot-state-url",
            BotConnectingState::ModeMismatch => "bot-state-mode",
            BotConnectingState::Timeouted => "bot-state-timeouted",
            BotConnectingState::Rejected => "bot-state-rejected",
            BotConnectingState::Incompatible => "bot-state-incompatible",
        }
    }
}

#[derive(Resource, Deref, DerefMut)]
struct BotConnecting(WebSocketConnector);
/// run if resource exists, otherwise not in connecting phase.
fn handle_connecting(
    mut ca: ResMut<BotConnecting>,
    mut commands: Commands,
    mut conn_state: ResMut<BotConnectingState>,
    mut timer: ResMut<ConnectingTimeout>,
) {
    match ca.step_connect() {
        Ok(Some(socket)) => {
            let mut conn = BotConnection(socket);
            if let Err(err) = conn.write(NotInGame::Discovery) {
                warn!("Writing to ws failed: {:?}", err);
                *conn_state = BotConnectingState::ConnFailed;
                return;
            }
            *conn_state = BotConnectingState::WaitingDiscovery;
            commands.insert_resource(conn);
            commands.remove_resource::<BotConnecting>();
            timer.reset();
        }
        Ok(None) => {}
        Err(WsError::Closed | WsError::Failed | WsError::AlreadyFailed)
        | Err(WsError::ParseError(_)) => {
            warn!("Connecting to Bot failed.");
            commands.remove_resource::<BotConnecting>();
            *conn_state = BotConnectingState::ConnFailed;
        }
        Err(WsError::Url) => {
            warn!("Connecting to Bot failed. (url)");
            commands.remove_resource::<BotConnecting>();
            *conn_state = BotConnectingState::UrlError;
        }
    }
}

/// run if not in_game and resource exists, otherwise not in bot mode.
fn not_in_game_handler(
    mut commands: Commands,
    mut ws: ResMut<BotConnection>,
    config: Res<GameConfig>,
    campaign_config: Res<CampaignModeConfig>,
    mut conn_lost_event: EventWriter<ConnectionLostEvent>,
    mut conn_state: ResMut<BotConnectingState>,
    mut timer: ResMut<ConnectingTimeout>,
) {
    match ws.read::<NotInGameResponse>() {
        Ok(Some(msg)) => {
            match msg {
                NotInGameResponse::Identity(bot_identity) => 'break_here: {
                    if *conn_state != BotConnectingState::WaitingDiscovery {
                        warn!("Bot send an unexpected Identity Package.");
                        return;
                    }
                    timer.reset();
                    let target_mode = match &config.selected_gamemode {
                        GameMode::Campaign => match campaign_config.level.get_bot_type() {
                            Some(CampaignBotType::Creative) => &GameMode::Creative,
                            Some(CampaignBotType::Challenge) => &GameMode::Challenge,
                            _ => {
                                // unsupported gamemode
                                // should be unreachable as we check it in the Ui
                                warn!(
                                    "Tried to connect to bot that provides {:?}, however the campaign level doesn't support bots at all",
                                    bot_identity.gamemode
                                );
                                *conn_state = BotConnectingState::ModeMismatch;
                                commands.remove_resource::<BotConnection>();
                                break 'break_here;
                            }
                        },
                        // zen mode behaves like challenge mode
                        // so we can allow challenge bots
                        GameMode::Zen => &GameMode::Challenge,
                        s => s,
                    };
                    if target_mode != &bot_identity.gamemode {
                        warn!(
                            "Tried to connect to a {:?} Bot but Bot only provides {:?}",
                            target_mode, bot_identity.gamemode
                        );
                        *conn_state = BotConnectingState::ModeMismatch;
                        commands.remove_resource::<BotConnection>();
                        break 'break_here;
                    }

                    if let Err(err) = ws.write(NotInGame::GameRequest) {
                        warn!("Writing to ws failed: {:?}", err);
                        *conn_state = BotConnectingState::ConnFailed;
                        commands.remove_resource::<BotConnection>();
                    }

                    *conn_state = BotConnectingState::WaitingConn;
                }
                NotInGameResponse::AcceptGame => {
                    if *conn_state != BotConnectingState::WaitingConn {
                        warn!("Bot send an unexpected AcceptGame Package.");
                        return;
                    }
                    timer.reset();
                    let target_mode = match &config.selected_gamemode {
                        GameMode::Campaign => match campaign_config.level.get_bot_type() {
                            Some(CampaignBotType::Creative) => &GameMode::Creative,
                            Some(CampaignBotType::Challenge) => &GameMode::Challenge,
                            // the connection is already aborted in NotInGameResponse::Identity
                            // when the campaign level doesn't support bots
                            _ => unreachable!(),
                        },
                        // zen mode behaves like challenge mode
                        // so we can allow challenge bots
                        GameMode::Zen => &GameMode::Challenge,
                        s => s,
                    };
                    info!("Bot connected. Start game in mode: {target_mode:?}");

                    let mode = config.selected_gamemode.clone();
                    commands.queue(move |w: &mut World| {
                        match mode {
                            GameMode::Challenge => w.run_system_once(init_challenge_mode).unwrap(),
                            GameMode::Creative => w.run_system_once(init_creative_mode).unwrap(),
                            GameMode::Campaign => w.run_system_once(init_campaign_mode).unwrap(),
                            GameMode::Zen => w.run_system_once(init_zen_mode).unwrap(),
                        };
                        *w.resource_mut::<BotConnectingState>() = BotConnectingState::None;
                    });
                }
                NotInGameResponse::RejectGame => {
                    commands.remove_resource::<BotConnection>();
                    warn!("Game request rejected!");
                    *conn_state = BotConnectingState::Rejected;
                }
            };
        }
        Ok(None) => {}
        Err(WsError::Closed | WsError::Failed | WsError::AlreadyFailed | WsError::Url) => {
            warn!("While connecting to Bot, WebSocket connection suddenly closed or failed.");
            commands.remove_resource::<BotConnection>();
            conn_lost_event.send(ConnectionLostEvent);
            *conn_state = BotConnectingState::ConnFailed;
        }
        Err(WsError::ParseError(err)) => {
            warn!("Received ControlPackage could not get parsed: {err}");
            commands.remove_resource::<BotConnection>();
            *conn_state = BotConnectingState::Incompatible;
            conn_lost_event.send(ConnectionLostEvent);
        }
    };
}

#[derive(Resource, Default)]
pub struct BotSettings {
    // settings
    pub action_limit: Option<usize>,
    pub disable_on_action: bool,
}

#[derive(Resource)]
pub struct BotState {
    // state
    pub execute_actions: bool,
    pub registered_events: HashSet<BotEventKind>,
}

impl Default for BotState {
    fn default() -> Self {
        Self {
            execute_actions: true,
            registered_events: HashSet::new(),
        }
    }
}

/// Will be send when an action is received.
#[derive(Event)]
pub struct ActionEvent {
    /// At least one restricted Action was send.
    pub restricted: bool,
    /// An empty ActionList was send.
    pub empty: bool,
}

/// Will be send when bot looses connection.
#[derive(Event)]
pub struct ConnectionLostEvent;

/// Will be send if the bot sent bad data.
#[derive(Event)]
pub struct BotDataErrorEvent;

/// Will be send if the bot sends to many actions..
#[derive(Event)]
pub struct BotActionOverflowEvent;

/// Will be send if the bot sends data to late, or none.
#[derive(Event)]
pub struct BotTimeoutEvent;

#[derive(Resource, Deref, DerefMut)]
pub struct BotConnection(WebSocket);

#[derive(SystemParam)]
pub struct ActionEvents<'w> {
    pub redraw: EventWriter<'w, ManualRedrawEvent>,
    pub tile: EventWriter<'w, TileActionEvent>,
    pub reward: EventWriter<'w, RewardReqEvent>,
    pub market: EventWriter<'w, ConfigureMarketReq>,
    pub rgb: EventWriter<'w, ChangeRgbTileColor>,
    pub chameleon: EventWriter<'w, ChangeChameleonTile>,
    pub tile_status: EventWriter<'w, TileStatusReq>,
}

// run if in_game and resource exists, otherwise not in bot mode.
#[allow(clippy::too_many_arguments)]
pub fn in_game_handler(
    mut commands: Commands,
    mut ws: ResMut<BotConnection>,
    settings: Res<BotSettings>,
    mut state: ResMut<BotState>,
    mut action_events: ActionEvents,
    mut action_event: EventWriter<ActionEvent>,
    mut conn_lost_event: EventWriter<ConnectionLostEvent>,
    mut parse_err_event: EventWriter<BotDataErrorEvent>,
    mut bot_event: EventReader<BotEvent>,
    mut bot_action_overflow: EventWriter<BotActionOverflowEvent>,
) {
    for event in bot_event.read() {
        if event.needs_no_registration() || state.registered_events.contains(&event.to_kind()) {
            if let Err(err) = ws.write(event.clone()) {
                match err {
                    WsError::Closed | WsError::Failed | WsError::AlreadyFailed | WsError::Url => {
                        commands.remove_resource::<BotConnection>();
                        conn_lost_event.send(ConnectionLostEvent);
                        warn!("The connection to the Bot was lost.");
                    }
                    WsError::ParseError(_) => unreachable!(
                        "Received ParseError while sending data to the Bot. This is impossible."
                    ),
                }
            }
        }
    }

    for _ in 0..MAX_ACTION_LISTS_PER_TICK {
        match ws.read::<ActionList>() {
            Ok(Some(msg)) => {
                let mut ignored = false;
                // check count if challenge
                let mut action_limit = if state.execute_actions {
                    settings.action_limit.unwrap_or(usize::MAX)
                } else {
                    0
                };
                let mut includes_restricted = false;
                let empty = msg.is_empty();
                // execute actions
                for action in msg.iter() {
                    if action.is_restricted() {
                        includes_restricted = true;
                        if action_limit > 0 {
                            action_limit -= 1;
                        } else {
                            ignored = true;
                            continue;
                        }
                    }
                    debug!("Received action");
                    match action {
                        BotAction::RegisterEvent(evt) => {
                            state.registered_events.insert(*evt);
                        }
                        BotAction::UnRegisterEvent(evt) => {
                            state.registered_events.remove(evt);
                        }
                        BotAction::UnRegisterAllEvents => {
                            state.registered_events.clear();
                        }
                        BotAction::Redraw => {
                            action_events.redraw.send(ManualRedrawEvent);
                        }
                        BotAction::PlaceTile(tile) => {
                            action_events
                                .tile
                                .send(TileActionEvent(TileActionType::Place(tile.clone())));
                        }
                        BotAction::TakeTile(coord) => {
                            action_events
                                .tile
                                .send(TileActionEvent(TileActionType::Take(*coord)));
                        }
                        BotAction::CollectReward(coord) => {
                            action_events.reward.send(RewardReqEvent(*coord));
                        }
                        BotAction::ConfigureMarketplaceTile(data) => {
                            action_events.market.send(data.clone());
                        }
                        BotAction::ConfigureRgbTile(data) => {
                            action_events.rgb.send(data.clone());
                        }
                        BotAction::ConfigureChameleonTile(data) => {
                            action_events.chameleon.send(data.clone());
                        }
                        BotAction::GetTileStatus(coord) => {
                            action_events.tile_status.send(TileStatusReq(*coord));
                        }
                    };
                }
                if settings.disable_on_action && (includes_restricted || empty) {
                    state.execute_actions = false;
                }
                if ignored {
                    bot_action_overflow.send(BotActionOverflowEvent);
                    warn!("The Bot send to many actions. Some will be ignored.");
                }
                action_event.send(ActionEvent {
                    restricted: includes_restricted,
                    empty,
                });
            }
            Ok(None) => {
                break;
            }
            Err(WsError::Closed | WsError::Failed | WsError::AlreadyFailed | WsError::Url) => {
                commands.remove_resource::<BotConnection>();
                conn_lost_event.send(ConnectionLostEvent);
                warn!("The connection to the Bot was lost.");
                break;
            }
            Err(WsError::ParseError(err)) => {
                warn!("Received BotAction could not get parsed: {err}");
                parse_err_event.send(BotDataErrorEvent);
                continue;
            }
        };
    }
}

fn headless_housekeeping(
    conn_lost: EventReader<ConnectionLostEvent>,
    state: ResMut<BotConnectingState>,
    mut exit: EventWriter<AppExit>,
) {
    let mut error = false;
    if state.is_error_state() {
        error = true;
        // error: message send by ws handler
    }
    if !conn_lost.is_empty() {
        error = true;
        error!("Lost connection to Bot.");
    }
    if error {
        exit.send(AppExit::error());
    }
}
